<?php
/**
 * aLinks WordPress Plugin
 *
 * A WordPress plugin that automatically links keywords in your
 * blog post.
 *
 * @author    Sean Hickey <seanhickey@gmail.com>
 * @package    alinks
 * @license    GPL
 * @version    $Id: aLinksModules.php 44 2008-03-24 18:00:53Z sean $
 */
/**
 * aLinks modules class
 *
 * This class handles task related to keyphrase modules, such as finding
 * them in the modules directory, and creating them.
 */
class aLinksModules extends aLinksApp
{	
	/**
	 * The full path to the modules
	 * directory.
	 * @var string
	 */
	var $modpath;
	
	/**
	 * Array of loaded modules.
	 * @var array
	 */
	var $modules;
	
	/**
	 * aLinks general settings.
	 * @var array
	 */
	var $settings;
	
	/**
	 * Class constructor
	 *
	 */
	function aLinksModules() {
		$this->modpath = ALINKS_MODULES_PATH;
		$this->modurl = ALINKS_MODULES_URL;
		$this->settings = get_option('alinks_settings');
		$this->scanForMods();
	}
	
	/**
	 * Loads all modules in the modules directory
	 *
	 * Scans the modules directory for modules, and loads the valid
	 * modules that it finds.  Returns TRUE if it loaded modules,
	 * or FALSE if it didn't.
	 *
	 * @return	 bool
	 */
	function scanForMods() {

		if (!is_admin()) {
			$dirs = get_option('alinks_module_dirs');
			if (empty($dirs)) {
				$dirs = $this->getDirs($this->modpath);
			}
		} else {
			$dirs = $this->getDirs($this->modpath);
			update_option('alinks_module_dirs', $dirs);
		}

		if (empty($dirs)) {
			return false;
		}

		foreach ($dirs as $dir) {
			$this->load($dir);
		}
		return true;
	}
	
	/**
	 * Loads a module
	 *
	 * Used to include and create the specified module object.  Returns
	 * TRUE if the module was succesfully loaded, or FALSE if it
	 * wasn't.
	 *
	 * @param	string	$mod	The module's directory name
	 * @return	bool
	 */
	function load($mod) {

		// - Make sure the mod.php script exists
		if (!is_readable($this->modpath . '/' . strtolower($mod) . '/mod.php')) {
			return false;
		}
		
		// - Make sure the mod.xml file exists
		if (!is_readable($this->modpath . '/' . strtolower($mod) . '/mod.xml')) {
			return false;
		}
		
		if (file_exists($this->modpath . '/' . strtolower($mod) . '/help.html')) {
			$helpHTML = $this->modurl . '/' . strtolower($mod) . '/help.html';
		}
		
		// - Get the XML contents and parse into an array
		$xml = file_get_contents($this->modpath . '/' . strtolower($mod) . '/mod.xml');
		if (empty($xml)) {
			return false;
		}
		$alinksXml = new aLinksXml;
		$alinksXml->setXML($xml);
		$modOptions = $alinksXml->parse();
		
		// - Read the mod.php file and find the name of the class using PHP's token_get_all
		$contents = file_get_contents($this->modpath . '/' . strtolower($mod) . '/mod.php');
		$tokens = token_get_all($contents);
		$class = null;
		for ($i = 0; $i < count($tokens); $i++) {
			if ($tokens[$i][0] == T_CLASS) {
			
				// - The actual name of the class will be 2 elements past the T_CLASS constant
				$class = $tokens[$i + 2][1];
				break;
			}
		}
		if (!$class) {
			return false;
		}
		
		// - Now that we have the class name, include the mod.php script
		include_once($this->modpath . '/' . strtolower($mod) . '/mod.php');
		if (!class_exists($class)) 	{
			return false;
		}
		
		// - Create the class object and pass it's, and aLink's, saved options to it
		$obj = new $class;
		if (!is_object($obj)) {
			return false;
		}
		$name = $obj->getName();
		$obj->setOptions($this->getModOptions($name));
		$obj->setSettings($this->settings);
		
		
		// - Now add the module to the $modules property array
		$this->modules[$name]['obj'] = &$obj;
		$this->modules[$name]['options'] = $modOptions['options'];
		$this->modules[$name]['defines'] = $modOptions['defines'];
		$this->modules[$name]['info'] = $modOptions['info'];
		$this->modules[$name]['dir'] = $mod;
		if (isset($helpHTML)) {
			$this->modules[$name]['help'] = $helpHTML;
		}
		
		return true;
	}
	
	/**
	 * Returns modules array
	 *
	 * @return	array
	 */
	function getModules() {
		return $this->modules;
	}
	
	/**
	 * Returns module options
	 *
	 * Each module can have a set of default options stored
	 * in the database.  This method returns those options. Returns
	 * an array of options, or NULL if the module has no options
	 * saved.
	 *
	 * @param	string	$mod	The name of the module
	 * @return	mixed
	 */
	function getModOptions($mod) {
		return get_option('alinks_mod_' . strtolower(trim($mod)));
	}

	function getModule($modName) {
		return $this->modules[$modName];
	}
	
	/**
	 * Gets a list of directories in a top directory
	 *
	 * Scans through the given directory and returns an array of
	 * all sub directories.  Returns FALSE on failure.
	 *
	 * @param	string	$dir	The name of the directory to scan
	 * @return	mixed
	 */
	function getDirs($dir) {
		$starttm = microtime(true);
	    if(is_dir($dir)) {
			$dirlist = opendir($dir);
			while(($file = readdir($dirlist)) !== false) {
				if(is_dir($dir . '/' . $file) && $file != '.' && $file != '..') {
					$dirs[] = $file;
				}
			}
	        return $dirs;
		} else {
			return false;
			break;
		}
	}
}