<?php
/**
 * aLinks WordPress Plugin
 *
 * A WordPress plugin that automatically links keywords in your
 * blog post.
 *
 * @author    Sean Hickey <seanhickey@gmail.com>
 * @package    alinks
 * @license    GPL
 * @version    $Id: aLinksModel.php 44 2008-03-24 18:00:53Z sean $
 */
/**
 * aLinks model class
 *
 * This class handles all of the DB input/output.
 */
class aLinksModel extends aLinksApp {

	/**
	 * The WPDB object.
	 * @var obj
	 */
	var $wpdb;
	
	/**
	 * The string that prefixes all WordPress
	 * tables.
	 * @var string
	 */
	var $table_prefix;
	
	/**
	 * The aLinks' general settings.
	 * @var array
	 */
	var $settings;
	
	/**
	 * Results from the last log file query.
	 * @var mixed
	 */
	var $logQueryResults = null;
	
	/**
	 * Class constructor
	 *
	 * @param	obj		$wpdb		The WPDB object
	 * @param	string	$table_prefix The string that prefixes all WP database tables
	 * @return	void
	 */
	function aLinksModel(&$wpdb, $table_prefix, &$alinksStruct) {
		
		$this->extractTo($this, $alinksStruct->getAll());
		$this->wpdb = $wpdb;
		$this->table_prefix = $table_prefix;
		$this->alinksLang = & new aLinksLang($this->alinks_languages_path);
		$this->settings = $this->getAllSettings();
		$this->makeTables();
		$this->doFileData($_FILES, $_POST);
		$this->doPostData($_POST, $_FILES);
		$this->doGetData($_GET);
	}
	
	/**
	 * Process uploaded files
	 *
	 * This method is automatically called by the constructor if there is $_FILES data.
	 * This will be used to process uploaded modules and imported keyphrases.  Returns
	 * FALSE if there was no files to process, or there was an error processing the
	 * file.  Returns TRUE if a file was processed.
	 *
	 * @param	array	$files	The $_FILES array
	 * @return	bool
	 */
	function doFileData($files, $post) {
		if (empty($files)) {
			return false;
		}
		if (empty($post['alinks_file'])) {
			return false;
		}
		
		if(get_magic_quotes_gpc() || get_magic_quotes_runtime()) {
			$post = $this->stripSlashesDeep($post);
		}
		
		$file = $post['alinks_file'];
		switch ($file) {
			case 'module':
				$this->notifyObjects('aLinksModel::doFileData', 'Installing-Module');
					require_once(ALINKS_CLASSES_PATH . '/pclzip.lib.php');
					$archive = new PclZip($files['modulefile']['tmp_name']);
					if (!is_writable(ALINKS_MODULES_PATH)) {
						$this->addSessionMessage($this->alinksLang->_('Error: Modules directory does not have write permissions.'), true);
						return false;
					}
					if ($archive->extract(PCLZIP_OPT_PATH, ALINKS_MODULES_PATH) == 0) {
						$this->addSessionMessage($this->alinksLang->_('Error: Could not open ZIP file.' . $archive->errorInfo(true)), true);
					} else {
						$this->addSessionMessage($this->alinksLang->_('Keyphrase module installed.'));	
					}
					$this->notifyObjects('aLinksModel::doFileData', 'Module-Installed');
				break;
		}
		return true;
	}

	/**
	 * Saves form data to the database
	 *
	 * This method is automatically called by the constructor if there is $_POST data
	 * available.  It takes data from a form and automatically stores that form data
	 * to the database.  This method always returns TRUE.
	 *
	 * @param	array	$post	The $_POST data
	 * @return	bool
	 */
	function doPostData($post, $files) {
		if (empty($post['alinks_action'])) {
			return false;
		}
		
		if(get_magic_quotes_gpc() || get_magic_quotes_runtime()) {
			$post = $this->stripSlashesDeep($post);
		}
		
		switch ($post['alinks_action']) {
			case 'settings':
				$this->saveSettings($post);
				break;
			case 'keyphrase':
				$this->saveKeyphrase($post);
				break;
			case 'export':
				$alinksImportExport = new aLinksImportExport($this->wpdb, $this->table_prefix);
				$alinksImportExport->export($post);
				break;
			case 'import':
				$alinksImportExport = new aLinksImportExport($this->wpdb, $this->table_prefix);
				$alinksImportExport->import($files, $post);
				break;
			case 'query':
				$this->queryLogs($post);
				break;
		}
		return true;
	}

	/**
	 * Called by the constructor if there is $_GET data
	 *
	 * This method is called automatically by the constructor if there is $_GET data
	 * available *besides* the page element.  Almost every page in the admin panel has
	 * $_GET data because the 'page' element is usually set.  So the constructor will only
	 * call this method is there is *other* $_GET data besides the page element.
	 * The $_GET data is passed to the method minus the page element.  Returns FALSE if
	 * the 'alinks_action' element was not set, or some other error.  Returns TRUE if
	 * the 'alinks_action' element was set, and the $_GET data was processed.
	 *
	 * @param	array	$get		The $_GET data
	 * @return	bool
	 */
	function doGetData($get) {
		if (empty($get['alinks_action'])) {
			return false;
		}

		if(get_magic_quotes_gpc() || get_magic_quotes_runtime()) {
			$get = $this->stripSlashesDeep($get);
		}

		$action = $get['alinks_action'];
		switch ($action) {
			case 'delete':
					$moduleDir = $this->alinks_modules_path .'/' . urldecode($get['module']);
					$this->removeDir($moduleDir);
					$this->addSessionMessage($this->alinksLang->_('Keyphrase module deleted.'));
					$this->addSessionRedirect('admin.php?page=alinks-Modules');
					$this->notifyObjects('aLinksModel::doGetData', 'Module-Deleted::' . $get['module']);
				break;
		}
		return true;
	}

	/**
	 * Returns aLinks' general settings
	 *
	 * Returns aLinks' general settings, or default settings if the general settings
	 * haven't been set.
	 *
	 * @return	array
	 */
	function getAllSettings() {
		$settings = get_option('alinks_settings');
		if (empty($settings)) {
			return $this->defaultSettings();
		}
		return $settings;
	}
	
	/**
	 * Sets default aLinks general settings
	 *
	 * Sets default aLinks general settings in the database, and returns those default
	 * general settings.
	 *
	 * @param	array
	 */
	function defaultSettings() {
		$defaultData = array	(
						'linkimage' => $this->alinks_url . '/images/external.png',
						'newwindow' => null,
						'clicktracking' => 'on',
						'customhtmljs' => null,
						'cssclass' => 'alinks_links',
						'maxlinks' => 1,
						'casesensitive' => null
							);
		update_option('alinks_settings', $defaultData);
		return $defaultData;
	}

	/**
	 * Saves general settings from Settings sub menu
	 *
	 * Runs through all the $_POST data from the general settings form, and
	 * saves the values in the wp_options table.
	 *
	 * @param	array	$post	The $_POST data
	 * @return	bool
	 */
	function saveSettings($post) {

		$nArray = array();
		$settingName = $post['setting_name'];
		unset($post['setting_name']);
		unset($post['alinks_action']);

		foreach ($post as $key => $value) {
			$nArray[$key] = $value;
		}
		update_option($settingName, $nArray);
		return true;
	}
	
	/**
	 * Returns the value of a general setting
	 *
	 * Returns the given setting for the key passed to the method.  Returns FALSE if
	 * the setting does not exists.
	 *
	 * @param	string	$key		The setting key
	 * @return	mixed
	 */
	function getSetting($key) {
		if (!array_key_exists($key, $this->settings)) {
			return false;
		}
		return $this->settings[$key];
	}
	
	/**
	 * Processes a form for a new keyphrase and saves the keyphrase
	 *
	 * This method is called when a form is submitted via the POST method, and it
	 * is a form for a new keyphrase.  The method will get all the values from the
	 * form, and then pass them to the newKeyphrase() method.  Returns 0 if the keyphrase
	 * was not added to the database, or 1 if it was.
	 *
	 * @param	array	$post	$_POST data from the new keyphrase form
	 * @return	int
	 */
	function saveKeyphrase($post) {
		if (empty($post['keyphrase'])) {
			$this->addSessionMessage('Keyphrase must be set', true);
			return false;
		}

		$keyphrase = $post['keyphrase'];
		$description = $post['description'];
		$module = strtolower($post['module']);
		echo $module;
		unset($post['keyphrase']);
		unset($post['description']);
		unset($post['module']);
		unset($post['alinks_action']);
		$defines = serialize($post);
		
		return $this->newKeyphrase($keyphrase, $description, $defines, $module);
	}
	
	/**
	 * Adds a new keyphrase to the database
	 *
	 * Takes all the required values for a new keyphrase, and saves them as a new row in the
	 * keyphrases table.  This method also creates the unique hash required for each
	 * keyphrase.  Returns 0 if the keyphrase was not added to the database, or 1 if it was.
	 *
	 * @param	string	$keyphrase	The new keyphrase
	 * @param	string	$description	The description of the keyphrase
	 * @param	string	$defines		A serialized array of the keyphrase's defines
	 * @param	string	$module		The name of the module that controls the keyphrase
	 * @return	int
	 */
	function newKeyphrase($keyphrase, $description, $defines, $module) {
		$keyphrase = 		$this->wpdb->escape($keyphrase);
		$description = 	$this->wpdb->escape($description);
		$defines = 		$this->wpdb->escape($defines);
		$module = 		$this->wpdb->escape($module);
		$hash = 			md5($keyphrase . $description . $defines . $module);
		$this->notifyObjects('aLinksModel::newKeyphrase', 'New-Keyphrase::' . $keyphrase . '::' . $module);
		return $this->wpdb->query	("
							INSERT INTO `{$this->table_prefix}alinks_keyphrases` 
							VALUES (
							null, 
							'$keyphrase', 
							'$description', 
							'$defines', 
							'$module', 
							null, 
							'$hash'
							)
								");
	}
	
	/**
	 * Returns the module used by a keyphrase
	 *
	 * Given the ID of a keyphrase, the method returns the type of module used by
	 * that keyphrase.  Returns NULL if the keyphrase doesn't exists.
	 *
	 * @param	int	$id	The keyphrase ID
	 * @return	mixed
	 */
	function getModuleType($id) {
		return $this->wpdb->get_var("SELECT module FROM {$this->table_prefix}alinks_keyphrases WHERE id = $id LIMIT 1");
	}
	
	/**
	 * Returns a single keyphrase's data
	 *
	 * Given the ID of a keyphrase, the method returns all the data for that keyphrase that
	 * is in the database.  Returns NULL if the keyphrase does not exists.
	 *
	 * @param	int	$id	The keyphrase's ID
	 * @return	mixed
	 */
	function getKeyphraseById($id) {
		return $this->wpdb->get_row("SELECT * FROM {$this->table_prefix}alinks_keyphrases WHERE id = $id LIMIT 1", ARRAY_A);
	}
	
	/**
	 * Returns an array of all defined keyphrases
	 *
	 * Returns a list of all defined keyphrases, if $searchTerm is specified, returns a list of
	 * defined keyphrases that match the search term.  Returns NULL if there are no keyphrases.
	 *
	 * @param	string	$searchTerm	The string to search for in keyphrases
	 
	 * @return	mixed
	 */
	function getKeyphraseList($searchTerm = null, $rand = false) {
		if (!empty($searchTerm)) {
			$searchTerm = $this->wpdb->escape($searchTerm);
			if ($rand) {
				$query = "SELECT * FROM {$this->table_prefix}alinks_keyphrases WHERE keyphrase LIKE '%$searchTerm%' ORDER BY RAND()";
			} else {
				$query = "SELECT * FROM {$this->table_prefix}alinks_keyphrases WHERE keyphrase LIKE '%$searchTerm%'";
			}
			return $this->wpdb->get_results($query, ARRAY_A);
		}
		if ($rand) {
			$query = "SELECT * FROM {$this->table_prefix}alinks_keyphrases ORDER BY RAND()";
		} else {
			$query = "SELECT * FROM {$this->table_prefix}alinks_keyphrases";
		}
		return $this->wpdb->get_results($query, ARRAY_A);
	}
	
	/**
	 * Find and replace for keyphrases
	 *
	 * Will replace any string specified by $find with the value of $replacement, only in the
	 * keyphrases.  Returns the number of rows affected.
	 *
	 * @param	string	$find		The string to find in the keyphrases
	 * @param	string	$replacement	The string to replace the find with
	 * @return	int
	 */
	function findAndReplace($find, $replacement) {
		if (empty($find) || empty($replacement)) {
			return false;
		}
		
		$find = $this->wpdb->escape($find);
		$replacement = $this->wpdb->escape($replacement);
		$this->notifyObjects('aLinksModel::findAndReplace', 'Find::' . $find . '::Replace::' . $replacement);
		return $this->wpdb->query("UPDATE `{$this->table_prefix}alinks_keyphrases` SET `keyphrase` = REPLACE(`keyphrase`, '$find', '$replacement')");
	}
	
	/**
	 * Updates an already existing keyphrase
	 *
	 * Updates the values for a keyphrase that already exists.  Only the keyphrase, the description, and
	 * the defines can be updated.  The module and hash cannot be changed once a keyphrase has been
	 * created.  Returns 1 if the keyphrase was updated, or 0 on failure.
	 *
	 * @param	int		$keyphraseID	The ID for the keyphrase to update
	 * @param	string	$keyphrase	The new keyphrase
	 * @param	string	$description	The new description
	 * @param	string	$defines		The new defines
	 * @return	int
	 */
	function updateKeyphrase($keyphraseID, $keyphrase, $description, $defines) {
		$this->notifyObjects('aLinksModel::updateKeyphrase', 'Update-Keyphrase::' . $keyphraseID);
		$keyphrase = $this->wpdb->escape($keyphrase);
		$description = $this->wpdb->escape($description);
		$defines = $this->wpdb->escape($defines);
		return $this->wpdb->query	("
							UPDATE `{$this->table_prefix}alinks_keyphrases` 
							SET `keyphrase` = '$keyphrase', 
							`description` = '$description', 
							`defines` = '$defines' 
							WHERE id = $keyphraseID
								");
	}
	
	/**
	 * Delets a keyphrase
	 *
	 * Deletes a keyphrase for a given ID.  Returns 1 if the keyphrase was deleted, or 0 if it
	 * was not.
	 *
	 * @param	int	$keyphraseID	The ID for the keyphrase to delete
	 * @return	int
	 */
	function deleteKeyphrase($keyphraseID) {
		$this->notifyObjects('aLinksModel::deleteKeyphrase', 'Delete-Keyphrase::' . $keyphraseID);
		return $this->wpdb->query("DELETE FROM `{$this->table_prefix}alinks_keyphrases` WHERE id = $keyphraseID");
	}
	
	function massDeleteKeyphrase($keyphrase) {
		$this->notifyObjects('aLinksModel::massDeleteKeyphrase', 'Mass-Delete-Keyphrases::' . $keyphrase);
		return $this->wpdb->query("DELETE FROM `{$this->table_prefix}alinks_keyphrases` WHERE keyphrase = '$keyphrase'");
	}
	
	/**
	 * Checks if an admin username and password are correct
	 *
	 * Returns TRUE if the given username and password are valid, and returns FALSE if they are
	 * not.
	 *
	 * @param	string	$username	The username to check
	 * @param	string	$password	The password for the username
	 * @return	bool
	 */
	function checkUnPw($username, $password) {
		if (!user_pass_ok($username, $password)) {
			return false;
		}
		return true;
	}

	/**
	 * Sends a query to the aLinksLogger object
	 *
	 * Takes a query from the Statistics form, and sends it to the aLinksLogger class object.  The
	 * results of the query are stored in the $logQueryResults propertry, and the generated HTML
	 * for the results are stored in the $logQueryResultsHtml property.  Returns FALSE if the query
	 * element doesn't exists in the $_POST data, or TRUE if the query was successfully sent to the
	 * aLinksLogger object.  A return value of TRUE does not mean the query itself was successful.
	 *
	 * @param	array	$post	The $_POST data
	 * @return	bool
	 */
	function queryLogs($post) {
		if (empty($post['query'])) {
			return false;
		}
		$alinksLogger = new aLinksLogger($this->alinks_log_path);
		$this->logQueryResults = $alinksLogger->query($post['query'], true);
		$this->logQueryResultsHtml = $alinksLogger->html;
		unset($alinksLogger);
		return true;
	}
	
	/**
	 * Returns the results from the last query to the aLinksLogger object
	 *
	 * @return	mixed
	 */
	function getLastLogQuery() {
		return $this->logQueryResults;
	}
	
	/**
	 * Returns the HTML generated by the last query to the aLinksLogger object
	 *
	 * @return	string
	 */
	function getLastLogQueryHtml() {
		return $this->logQueryResultsHtml;
	}
	
	/**
	 * Creates the keyphrases database table if it doesn't exists
	 *
	 * Checks to see if the wp_alinks_keyphrases table exists in the database, and if it
	 * doesn't, creates the table.
	 *
	 * @return	void
	 */
	function makeTables() {
		$result = $this->wpdb->get_results("SHOW TABLES LIKE '{$this->table_prefix}alinks_keyphrases'", ARRAY_A);
		if ($result) {
			return;
		}
		require_once(ABSPATH . 'wp-admin/upgrade-functions.php');
		$sql = "
		CREATE TABLE {$this->table_prefix}alinks_keyphrases (
		id int(11) NOT NULL AUTO_INCREMENT,
		keyphrase varchar(250) NOT NULL,
		description varchar(250) NOT NULL,
		defines text NOT NULL,
		module varchar(60) NOT NULL,
		created timestamp NOT NULL,
		hash varchar(32) NOT NULL,
		PRIMARY KEY  (id),
		KEY keyword_keyword (keyphrase)
		);";
		dbDelta($sql);
		
		// - Upgrade from old version
		$oldGeneral = get_option('autolinks_keywords');
		$oldAmazon = get_option('autolinks_amazonkeywords');
		if (empty($oldGeneral) && empty($oldAmazon)) {
			return;
		}
		
		if (!empty($oldGeneral)) {
			foreach ($oldGeneral as $keyword => $url) {
				$defines = array('url' => $url);
				$sDefines = serialize($defines);
				$this->newKeyphrase($keyword, '', $sDefines, 'General');
			}
			delete_option('autolinks_keywords');
		}
		
		if (!empty($oldAmazon)) {
			foreach ($oldAmazon as $searchterm => $producttype) {
				$defines = array('searchterm' => $searchterm, 'producttype' => $producttype);
				$sDefines = serialize($defines);
				$this->newKeyphrase($searchterm, '', $sDefines, 'Amazon Basic');
			}
			delete_option('autolinks_amazonkeywords');
		}
	}
}