<?php

/* ===========================================================================*/
//
// This code is provided free on the basis that you do not claim that
// it is your own, sell it or use it as the basis for other products that you
// sell. By all means extend it, modify it, upgrade it, correct it,
// suggest improvements, call me an idiot, etc.
//
// (c) 2004/09
// Andrew Dearing
// European Industrial Research Management Association
// www.eirma.org
//
// v2.3.0, 10.06.2009
// For VB3.7.x and VB3.8.x and VB3.8.x
// see changes.txt for history
// v1.00, 1.3.2004
//
/* ===========================================================================*/

// -----------------------------------------------------------------------------
// Entity handling functions
//
// Entities provide a means to associate arbitrary data with LDM entries. They
// are held as records in the entity table. Each value is uniquely associated
// with one LDM entry and is contained in one entity group.
//
// A data dictionary is stored as the serialised record [entity_definitions] in
// the admin table. It contains information on entry identifiers, the groups of
// which they are members and the order in which entries and groups
// are displayed inside linkbits. The dictionary is inheritable by categories
// in the usual way, although only some values are exposed for customisation.
//
// Information for plug-in authors:
//
//   1. call ldm_define_entity() to include entity in the data dictionary and define its group, type and
//      display order.  If the entity is already defined, call has no effect
//
//   2. call ldm_insert_entity() to associate an entity value with an entry
//
//   3. call ldm_delete_entities() to delete one/all entity values associated with given entry
//
//   4. call ldm_force_entity_consistency() to ensure consistency between data dictionary and entity tables
//
// -----------------------------------------------------------------------------

/**
* Define standard data structures
*/

function ldm_standard_entity(&$entities, &$group, &$entity) {

	$entities = array(
		"groups"=>array(),
		"entities"=>array()
		);

	$group = array(
		'entitygroupid' => 0,
		'entitygroup' => "",
		'entitygrouporder' => 1,
		'entitygrouphidden' => 0,
		'entitygroupcollapse' => 0,
		'entitycount' => 0
		);

	$entity = array(
		'entityname' => "",
		'entitytype' => ENTITY_TYPE_TEXT,
		'entitygroup' => "",
		'displayorder' => 1,
		'required' => 0,
		'entitycount' => 0,
		);

}

/**
* Add new entity definition to data dictionary
*
* @param	str		Entity name
* @param	str		Entity group
* @param	str		Entity type
* @param	int		Display order, negative values will be hidden from standard users
* @return			None
*/

function ldm_define_entity($entityname, $entitygroup, $entitytype=ADMIN_ENTITY_TEXT, $displayorder=-1) {
	global $vbulletin;
	global $links_defaults;

	$entity_definitions = unserialize(ldm_lookup_setting(-1, 'entity_definitions'));

	$entityexists = 0;
	foreach ($entity_definitions['entities'] as $thisid=>$thisentity) {
		if ($thisentity['entityname']==$entityname) {
			$entityexists = 1;
			break;
		}
	}

	if (!$entityexists) {

		$entity_definitions['entities'][] = array(
			'entityname'=>$entityname,
			'entitytype'=>intval($entitytype),
			'entitygroup'=>$entitygroup,
			'displayorder'=>$displayorder,
		);
		uasort($entity_definitions['entities'], 'ldm_entity_sort');

		$groupexists = 0;
		foreach ($entity_definitions['groups'] as $thisid=>$thisgroup) {
			if ($thisgroup['entitygroup']==$entitygroup) {
				$groupexists = 1;
				break;
			}
		}

		if (!$groupexists) {
			$nextkey = 0;
			if (count($entity_definitions['groups'])) {
				$nextkey = max(0, max(array_keys($entity_definitions['groups']))+1);
			}
			$entity_definitions['groups'][$nextkey] = array(
				'entitygroupid' => $nextkey,
				'entitygroup' => $entitygroup,
				'entitygrouporder' => $displayorder,
				'entitycount' => 1,
				);
			uasort($entity_definitions['groups'], 'ldm_entitygroup_sort');
		}

		ldm_update_setting(-1, 'entity_definitions', serialize($entity_definitions));

	}

}

/**
* Insert entities associated with given entryid
*
* @param	int		Entry id
* @param	arr		Entity
* @return			None
*/

function ldm_insert_entities($linkid, $entities) {
	global $vbulletin;

	if (!count($entities)) return;

	$entity_inserts = array();
	foreach ($entities as $thisentity) {
		$entity_inserts[] = "(" .
			"'" . intval($linkid) . "', " .
			"'" . $vbulletin->db->escape_string($thisentity['entityname']) . "', " .
			"'" . $vbulletin->db->escape_string($thisentity['entitydesc']) . "', " .
			"'" . $vbulletin->db->escape_string($thisentity['entityvalue']) . "', " .
			"'" . intval($thisentity['entitytype']) . "', " .
			"'" . TIMENOW . "', " .
			"'" . $vbulletin->db->escape_string($thisentity['entitygroup']) . "'".
			")";
	}

	$vbulletin->db->query_insert(
		THIS_TABLE."linksentities",
		'(linkid, entityname, entitydesc, entityvalue, entitytype, entitydate, entitygroup)',
		$entity_inserts
	);

}

/**
* Insert or update entities associated with given entryid
*
* @param	int		Entry id
* @param	arr		Entity
* @return			None
*/

function ldm_replace_entities($linkid, $entities) {
	global $vbulletin;

	if (!count($entities)) return;

	$query = "
		SELECT entityname
		FROM ". THIS_TABLE . "linksentities
		WHERE linkid='".$linkid."'
		";

	$asb = $vbulletin->db->query_read($query);
	$curr_entities = array();
	while ($rec=$vbulletin->db->fetch_array($asb)) {
		$curr_entities[] = $rec['entityname'];
	}

	$entity_inserts = array();
	foreach ($entities as $thisentity) {
		if (in_array($thisentity['entityname'], $curr_entities)) {
			$query = "
				UPDATE ". THIS_TABLE . "linksentities
				SET
					entitydesc='" . $vbulletin->db->escape_string($thisentity['entitydesc']) . "',
					entityvalue='" . $vbulletin->db->escape_string($thisentity['entityvalue']) . "',
					entitydate='" . TIMENOW . "'
				WHERE linkid='".intval($linkid)."'
				AND entityname='" . $vbulletin->db->escape_string($thisentity['entityname']) . "'
				";
			$asb = $vbulletin->db->query_write($query);
		}
		else {
			$entity_inserts[] = "(" .
				"'" . intval($linkid) . "', " .
				"'" . $vbulletin->db->escape_string($thisentity['entityname']) . "', " .
				"'" . $vbulletin->db->escape_string($thisentity['entitydesc']) . "', " .
				"'" . $vbulletin->db->escape_string($thisentity['entityvalue']) . "', " .
				"'" . intval($thisentity['entitytype']) . "', " .
				"'" . TIMENOW . "', " .
				"'" . $vbulletin->db->escape_string($thisentity['entitygroup']) . "'".
				")";
		}
	}

	if (count($entity_inserts)) {
		$vbulletin->db->query_insert(
			THIS_TABLE."linksentities",
			'(linkid, entityname, entitydesc, entityvalue, entitytype, entitydate, entitygroup)',
			$entity_inserts
		);
	}

}

/**
* Delete entities associated with given entryids
*
* @param	mixed	Entry ids
* @param	mixed	Optional, entity names
* @return			None
*/

function ldm_delete_entities($linkids, $entitynames=array()) {
	global $vbulletin;

	if (!is_array($linkids)) {
		$linkids = array($linkids);
	}

	if (!count($linkids)) {
		return;
	}

	$ids = implode(',', $linkids);

	if (!is_array($entitynames)) {
		$entitynames = array($entitynames);
	}

	$names = "";
	if (count($entitynames)) {
		foreach ($entitynames as $kid=>$kname) {
			$entitynames[$kid] = $vbulletin->db->escape_string($entitynames[$kid]);
		}
		$names = implode("', '", $entitynames);
	}

// Clean up any existing uploads...
	$asb = $vbulletin->db->query_read("
		SELECT entityid, entityvalue
		FROM ". THIS_TABLE . "linksentities
		WHERE entitytype IN (" . ENTITY_TYPE_UPLOAD .",". ENTITY_TYPE_MEDIAUPLOAD .",". ENTITY_TYPE_IMAGEUPLOAD .")
		AND linkid IN (" . $ids .")
		" . ($names ? "AND entityname IN ('".$names."')" : "") . "
		");
	while ($rec=$vbulletin->db->fetch_array($asb)) {
		ldm_delete_upload($rec['entityvalue']);
	}
	$vbulletin->db->free_result($asb);

	$query = "
		DELETE
		FROM ". THIS_TABLE . "linksentities
		WHERE linkid IN (".$ids.")
		" . ($names ? "AND entityname IN ('".$names."')" : "") . "
		";
	$asb = $vbulletin->db->query_write($query);
	$vbulletin->db->free_result($asb);

}

/**
* Fetch entity/entities for given entryid
*
* @param	int		Entry id
* @param	str		Optional, entity name
* @return	array	Entity array
*/

function ldm_fetch_entities($linkid, $entityname="") {
	global $vbulletin;

	$query = "
		SELECT *
		FROM ". THIS_TABLE . "linksentities
		WHERE linkid='".$linkid."'
		" . ($entityname ? "AND entityname='".$vbulletin->db->escape_string($entityname)."'" : "") . "
		" . ($entityname ? "" : "ORDER BY entityname") . "
		";
	$asb = $vbulletin->db->query_read($query);
	$entities = array();
	while ($rec=$vbulletin->db->fetch_array($asb)) {
		$entities[$rec['entityname']] = $rec;
	}
	$vbulletin->db->free_result($asb);

	return $entities;

}

/**
* Fetch entity counts
*
* @return	array	Entity count array
*/

function ldm_fetch_entity_count() {
	global $vbulletin;

	$query = "
		SELECT entityname, COUNT(linkid) AS entitycount
		FROM ". THIS_TABLE . "linksentities
		GROUP BY entityname
		";
	$asb = $vbulletin->db->query_read($query);
	$entities = array();
	while ($rec=$vbulletin->db->fetch_array($asb)) {
		$entities[$rec['entityname']] = $rec['entitycount'];
	}

	return $entities;

}

/**
* Deletes any entity display order overrides for given category
*
* @param	str		Category
* @return			None
*/

function ldm_clear_entity_displayorder($catid) {
	global $vbulletin;

	$query = "
		DELETE FROM ".THIS_TABLE."linksadmin
		WHERE settingname='entity_definitions'
		AND catid=".intval($catid)."
		";
	$vbulletin->db->query_write($query);

	ldm_datastore_markdirty('ldm_admin');
}

/**
* Define entity display orders for given category
*
* @param	str		Category
* @param	arr		Entity group collapse flags
* @param	arr		Entity group hidden flags
* @param	arr		Entity group display orders
* @param	arr		Entity display orders
* @return			None
*/

function ldm_set_entity_displayorder($catid, $entitygroupcollapse, $entitygrouphidden, $entitygrouporder, $entityorder, $entityrequired) {
	global $ldm_master_settings;

	$curr_entity_definitions = ldm_lookup_composite_entitydef($catid);

	if (isset($ldm_master_settings['entity_definitions'][$catid])) {
		$entity_definitions = unserialize($ldm_master_settings['entity_definitions'][$catid]['setting']);
	}
	else {
		ldm_standard_entity($entity_definitions, $group_def, $entity_def);
	}

// Per-category over-rides are limited to certain parts of the data structure
	foreach ($curr_entity_definitions['groups'] as $groupid=>$groupdef) {
		$collapse = (isset($entitygroupcollapse[$groupid]) ? $entitygroupcollapse[$groupid] : 0);
		if ($collapse != $curr_entity_definitions['groups'][$groupid]['entitygroupcollapse']) {
			$entity_definitions['groups'][$groupid]['entitygroupcollapse'] = $collapse;
		}
		$hidden = (isset($entitygrouphidden[$groupid]) ? $entitygrouphidden[$groupid] : 0);
		if ($hidden != $curr_entity_definitions['groups'][$groupid]['entitygrouphidden']) {
			$entity_definitions['groups'][$groupid]['entitygrouphidden'] = $hidden;
		}
		$displayorder = (isset($entitygrouporder[$groupid]) ? $entitygrouporder[$groupid] : 0);
		if ($displayorder != $curr_entity_definitions['groups'][$groupid]['entitygrouporder']) {
			$entity_definitions['groups'][$groupid]['entitygrouporder'] = $displayorder;
		}
	}

	foreach ($curr_entity_definitions['entities'] as $entityid=>$entitydef) {
		$displayorder = (isset($entityorder[$entityid]) ? $entityorder[$entityid] : 0);
		if ($displayorder != $curr_entity_definitions['entities'][$entityid]['displayorder']) {
			$entity_definitions['entities'][$entityid]['displayorder'] = $displayorder;
		}
		$required = (isset($entityrequired[$entityid]) ? $entityrequired[$entityid] : 0);
		if ($required !== $curr_entity_definitions['entities'][$entityid]['required']) {
			$entity_definitions['entities'][$entityid]['required'] = $required;
		}
	}

	if (count($entity_definitions['groups']) or count($entity_definitions['entities'])) {
		ldm_update_oneadminrow($catid, 'entity_definitions', serialize($entity_definitions));
	}

}

/**
* Handle the entity data on the Add/Edit Entry form
*
* @param	int		Entry id
* @param	array	Entity data
* @param	array	Entity files data for uploads
* @return			Bool, success/fail
*/

function ldm_associate_entities($linkid, $entities, $entityfiles, &$statusmessage) {
	global $vbulletin, $vbphrase;
	global $links_defaults;

// Process any uploads
	$entity_upload = array();
	$entity_uploadedfile = array();

	if (is_array($entityfiles)) {
		foreach ($entityfiles as $thisfield=>$thesefiles) {
			foreach ($thesefiles as $thisentityid=>$thisfieldval) {
				if ($thisfieldval) {
					$entity_upload[$thisentityid][$thisfield] = $thisfieldval;
				}
			}
		}
		foreach ($entity_upload as $thisentityid=>$thisupload) {
			if ($thisupload['name']) {
				$do_upload = ldm_process_upload($thisupload, $thisuploadedfile, $links_defaults['upload_filetypes'], $statusmessage);
				if ($do_upload) {
					$entity_uploadedfile[] = $entities[$thisentityid]['entityvalue'] = $thisuploadedfile;
				}
				else { // Clean up before returning...
					foreach ($entity_uploadedfile as $thisuploadedfile) {
						ldm_delete_upload($thisuploadedfile);
					}
					return 0;
				}
			}
		}
	}

// Pathway for handling uploads when swfuploader is active...
	foreach ($entities as $thisentityid=>$thisentity) {
		if (array_key_exists('uploadfile', $entities[$thisentityid])) {
			if (strlen($entities[$thisentityid]['uploadfile'])>0) {
				$entities[$thisentityid]['entityvalue'] = $entities[$thisentityid]['uploadfile'];
			}
		}
	}

// Find which values are set/changed
	foreach ($entities as $thisentityid=>$thisentity) {
		if (!array_key_exists('entityvalue', $entities[$thisentityid])) {
			$entities[$thisentityid]['entityvalue'] = $entities[$thisentityid]['currvalue'];
		}
		$entities[$thisentityid]['entityset'] =
			($entities[$thisentityid]['entityvalue']!=$entities[$thisentityid]['currvalue'] or
			$entities[$thisentityid]['entitydesc']!=$entities[$thisentityid]['currdesc']);
	}
	$entity_definitions = unserialize(ldm_lookup_setting(-1, 'entity_definitions'));

// Look up the entity definitions
	foreach ($entities as $thisentityid=>$thisentity) {
		unset($thisentitydef);
		foreach ($entity_definitions['entities'] as $thisdef) {
			if ($thisentity['entityname']==$thisdef['entityname']) {
				$thisentitydef = $thisdef;
				break;
			}
		}

// Trap attempts to insert an undefined entity...
		if (isset($thisentitydef)) {
			$entities[$thisentityid]['entitydef'] = $thisentitydef;
		}
		else {
			unset($entities[$thisentityid]);
		}
	}

	$allrequired = 1;
	$entityerrors = array();
	foreach ($entities as $thisentityid=>$thisentity) {
		if (!$thisentity['required']) {
			continue;
		}
		if ($thisentity['entitydelete']) {
			$allrequired = 0;
			continue;
		}
		$thisentitydef =& $thisentity['entitydef'];
		switch ($thisentitydef['entitytype']) {
		case ENTITY_TYPE_YESNO:
			if (!isset($thisentity['entityvalue'])) {
				$allrequired = 0;
				$entityerrors[] = $thisentity['entityname'];
			}
			break;
		default:
			if ($thisentity['entityvalue']==="") {
				$allrequired = 0;
				$entityerrors[] = $thisentity['entityname'];
			}
			break;
		}
	}

	if (!$allrequired) {
		$statusmessage = construct_phrase($vbphrase['ll_error_entity_value_required'], implode(', ', $entityerrors)) .
		    $vbphrase['ll_goback'];
		foreach ($entity_uploadedfile as $thisuploadedfile) {
			ldm_delete_upload($thisuploadedfile);
		}
		return 0;
	}

	$allvalidtype = 1;
	$entityerrors = array();
	foreach ($entities as $thisentityid=>$thisentity) {

		if ($thisentity['entityvalue']==="" or $thisentity['entitydelete']) {
			continue;
		}

		$thisentitydef =& $thisentity['entitydef'];

		switch ($thisentitydef['entitytype']) {

		case ENTITY_TYPE_IMAGE:
		case ENTITY_TYPE_IMAGEUPLOAD:
            require_once (DIR . '/includes/local_links_images.php');
            if (ldm_can_handle_as_image($thisentity['entityvalue'])<=0) {
            	$allvalidtype = 0;
				$entityerrors[] = $thisentity['entityname'];
            }
			break;

		case ENTITY_TYPE_MEDIA:
		case ENTITY_TYPE_MEDIAUPLOAD:
            require_once (DIR . '/includes/local_links_players.php');
			$urlInfo	= ldm_parse_url($thisentity['entityvalue']);
			$urlType	= file_extension($urlInfo['path']);
			$filetype   = strtolower($urlType);
			if (get_ldm_playerid($linkid, $thisentity['entityvalue'], $filetype)<0) {
            	$allvalidtype = 0;
				$entityerrors[] = $thisentity['entityname'];
			}
			break;

		default:
			break;
		}
	}

	if (!$allvalidtype) {
		$statusmessage = construct_phrase($vbphrase['ll_error_entity_value_type'], implode(', ', $entityerrors)) .
		    $vbphrase['ll_goback'];
		foreach ($entity_uploadedfile as $thisuploadedfile) {
			ldm_delete_upload($thisuploadedfile);
		}
		return 0;
	}

	$entity_delete = array();
	foreach ($entities as $thisentityid=>$thisentity) {
		if (!$thisentity['entityset'] and !$thisentity['entitydelete']) {
			unset($entities[$thisentityid]);
			continue;
		}
		$thisentitydef =& $thisentity['entitydef'];
		$entities[$thisentityid]['entitytype'] = $thisentitydef['entitytype'];
		$entities[$thisentityid]['entitygroup'] = $thisentitydef['entitygroup'];
		if ($thisentity['entitydelete']) {
			$entity_delete[] = $thisentity['entityname'];
			unset($entities[$thisentityid]);
		}

	}

	if (count($entity_delete)) {
		ldm_delete_entities($linkid, $entity_delete);
	}

	if (count($entities)) {
		ldm_replace_entities($linkid, $entities);
	}

	return 1;
}

/**
* Sort helper routines
*
*/

function ldm_entity_sort($a, $b) {
	if ($a['entitygroup'] == $b['entitygroup']) {
		if ($a['displayorder'] == $b['displayorder']) {
			return strcmp($a['entityname'], $b['entityname']);
		}
		else {
			return ($a['displayorder'] < $b['displayorder']) ? -1 : 1;
		}
	}
	else {
		return ($a['entitygroup'] < $b['entitygroup']) ? -1 : 1;
	}
}

function ldm_entitybit_sort($a, $b) {
	if ($a['linkid'] == $b['linkid']) {
		if ($a['entitygroup'] == $b['entitygroup']) {
			if ($a['displayorder'] == $b['displayorder']) {
				return strcmp($a['entityname'], $b['entityname']);
			}
			else {
				return ($a['displayorder'] < $b['displayorder']) ? -1 : 1;
			}
		}
		else {
			return strcmp($a['entitygroup'], $b['entitygroup']);
		}
	}
	else {
		return ($a['linkid'] < $b['linkid']) ? -1 : 1;
	}
}

function ldm_entitygroup_sort($a, $b) {
	if ($a['entitygrouporder'] == $b['entitygrouporder']) {
		if ($a['entitygroup'] == $b['entitygroup']) {
			return 0;
		}
		else {
			return ($a['entitygroup'] < $b['entitygroup']) ? -1 : 1;
		}
	}
	else {
		return ($a['entitygrouporder'] < $b['entitygrouporder']) ? -1 : 1;
	}
}

/**
* Set up the 'Add Attributes' bit
*
* @param	int		Entryid
* @param	bool	Show hidden entities
* @return	str		AddEntitybit
*/

function ldm_get_addentitybits($linkid, $catid, $showhidden=0) {
	global $vbulletin, $stylevar, $vbphrase, $vbcollapse;
	global $LINKS_SCRIPT;
	global $links_defaults, $links_permissions;
	global $LDM_environment;
	global $ldm_headtag_include;

	$entitybits = "";

	$ENTITY_TYPE_YESNO = ENTITY_TYPE_YESNO;
	$ENTITY_TYPE_TEXT = ENTITY_TYPE_TEXT;
	$ENTITY_TYPE_URL = ENTITY_TYPE_URL;
	$ENTITY_TYPE_MEDIA = ENTITY_TYPE_MEDIA;
	$ENTITY_TYPE_UPLOAD = ENTITY_TYPE_UPLOAD;
	$ENTITY_TYPE_MEDIAUPLOAD = ENTITY_TYPE_MEDIAUPLOAD;
	$ENTITY_TYPE_IMAGE = ENTITY_TYPE_IMAGE;
	$ENTITY_TYPE_IMAGEUPLOAD = ENTITY_TYPE_IMAGEUPLOAD;
	$ENTITY_TYPE_HIDDEN = ENTITY_TYPE_HIDDEN;

	$entity_definitions = ldm_lookup_composite_entitydef($catid);

	$linkentities = array();
	if (count($entity_definitions['entities']) and $linkid>0) {
		$query = "
			SELECT *
			FROM ". THIS_TABLE . "linksentities
			WHERE linkid = ".$linkid."
			";
		$asb = $vbulletin->db->query_read($query);
		while ($myrow = $vbulletin->db->fetch_array($asb)) {
			$linkentities[$myrow['entityname']] = array('entityvalue'=>$myrow['entityvalue'], 'entitydesc'=>$myrow['entitydesc']);
		}
	}

	asort($entity_definitions['groups']);

	foreach ($entity_definitions['groups'] as $entity_groupid=>$thisgroup) {

		$entity_group = $thisgroup['entitygroup'];

		$entity_group_translated = $entity_group;
		if ($entity_group[0]=="*") {
			$entity_phrase = substr($entity_group,1,strlen($entity_group)-1);
			if (isset($vbphrase[$entity_phrase])) {
				$entity_group_translated = $vbphrase[$entity_phrase];
			}
		}

		if ($thisgroup['entitygrouphidden']) {
			continue;
		}

		$hiddengroup = 0;
		if ($thisgroup['entitygrouporder']<0) {
			if (!$links_permissions['can_view_hidden']) {
				continue;
			}
			$hiddengroup = 1;
		}

		if (isset($vbcollapse['collapseobj_entitygroup'.$entity_groupid])) {
			$collapseobj = $vbcollapse['collapseobj_entitygroup'.$entity_groupid];
			$collapseimg = $vbcollapse['collapseimg_entitygroup'.$entity_groupid];
		}
		elseif ($thisgroup['entitygroupcollapse']) {
			$collapseobj = "display:none;";
			$collapseimg = "_collapsed";
		}
		else {
			$collapseobj = "";
			$collapseimg = "";
		}

		$entitybit = array();
		foreach ($entity_definitions['entities'] as $thisid=>$thisentity) {
			if ($thisentity['entitygroup']!=$entity_group) {
				continue;
			}
			$displayorder = $thisentity['displayorder'];
			if (($entity_type==$ENTITY_TYPE_HIDDEN or $displayorder<0) and !$links_permissions['can_view_hidden']) {
				continue;
			}
			$hidden = 0;
			if ($hiddengroup or $displayorder<0) {
				$hidden = 1;
			}
			$entity_type = $thisentity['entitytype'];
			$entity_name = $thisentity['entityname'];

			$entity_translated = $entity_name;
			if ($entity_name[0]=="*") {
				$entity_phrase = substr($entity_name,1,strlen($entity_name)-1);
				if (isset($vbphrase[$entity_phrase])) {
					$entity_translated = $vbphrase[$entity_phrase];
				}
			}

			$entity_required = $thisentity['required'];
			$entity_seq = $thisid;
			$entity_value = "";
			$entity_desc = "";
			if ($linkid and isset($linkentities[$entity_name])) {
				$entity_value = htmlspecialchars_uni($linkentities[$entity_name]['entityvalue']);
				$entity_desc = htmlspecialchars_uni($linkentities[$entity_name]['entitydesc']);
			}
			if ($entity_required and !$entity_value) {
				$collapseobj = "";
				$collapseimg = "";
			}

			eval('$oneentitybit = "' . fetch_template('links_addentity') . '";');

			($hook = vBulletinHook::fetch_hook('ldm_addentity_oneitem')) ? eval($hook) : false;

			$entitybit[$displayorder] .= $oneentitybit;
		}

		if (count($entitybit)) {
			ksort($entitybit);
			$entitybit = implode('', $entitybit);
			eval('$entitygroupbit = "' . fetch_template('links_addentitygroup') . '";');

			($hook = vBulletinHook::fetch_hook('ldm_addentity_onegroup')) ? eval($hook) : false;

			$entitybits .= $entitygroupbit;
		}
	}
	return $entitybits;
}

/**
* Construct a composite entity_definitions structure for given category
*
* @param	int		Category id
* @return	arr		Entity_definitions
*/

function ldm_lookup_composite_entitydef($catid) {
	global $linkscat, $ldm_master_settings, $BASE_CAT;

	ldm_standard_entity($entity_definitions, $entity_group, $entity);

	if (!isset($ldm_master_settings['entity_definitions'])) {
		return $entity_definitions;
	}

	if ($catid>0) {
		$cat_choice = array_reverse(explode(',', $catid . ',' . $linkscat[$catid]['parentlist']));
	}
	else {
		$cat_choice = array($BASE_CAT);
	}

// Start by taking the global values, then successively overwrite with child categories until reach this category
	foreach ($cat_choice as $c_catid) {
		if (isset($ldm_master_settings['entity_definitions'][$c_catid])) {
			$these_definitions = unserialize($ldm_master_settings['entity_definitions'][$c_catid]['setting']);
			foreach ($these_definitions['groups'] as $thisid=>$thisset) {
				if (!isset($entity_definitions['groups'][$thisid])) {
					$entity_definitions['groups'][$thisid] = $entity_group;
				}
				foreach ($thisset as $thisvalid=>$thisval) {
					$entity_definitions['groups'][$thisid][$thisvalid] = $thisval;
				}
			}
			foreach ($these_definitions['entities'] as $thisid=>$thisset) {
				if (!isset($entity_definitions['entities'][$thisid])) {
					$entity_definitions['entities'][$thisid] = $entity;
				}
				foreach ($thisset as $thisvalid=>$thisval) {
					$entity_definitions['entities'][$thisid][$thisvalid] = $thisval;
				}
			}
		}
	}

	return $entity_definitions;
}

/**
* Set up the 'Add/Edit Category' attributes bit
*
* @param	int		Category id
* @param	int		Include hidden attributes
* @return	str		Entitybit
*/

function ldm_get_catentitybits($catid, $showhidden=0) {
	global $links_defaults, $links_permissions, $linkscat, $ldm_master_settings;
	global $vbphrase, $stylevar;

	$entity_definitions = ldm_lookup_composite_entitydef($catid);
	$is_override = 0;

	if ($catid>0 and isset($ldm_master_settings['entity_definitions'][$catid])) { // overridden in this category
		$is_override = 1;
	}
	else {
		$parentlist = explode(",", $linkscat[$catid]['parentlist']);
		foreach ($parentlist as $thisparent) {
			if ($thisparent>0 and isset($ldm_master_settings['entity_definitions'][$thisparent])) { // overridden in a parent category
				$is_override = 2;
				break;
			}
		}
	}

	$entitybits = "";

	foreach ($entity_definitions['groups'] as $thisgroup) {

		$entity_group = $thisgroup['entitygroup'];

		$entity_group_translated = $entity_group;
		if ($entity_group[0]=="*") {
			$entity_phrase = substr($entity_group,1,strlen($entity_group)-1);
			if (isset($vbphrase[$entity_phrase])) {
				$entity_group_translated = $vbphrase[$entity_phrase];
			}
		}

		$entity_groupid = $thisgroup['entitygroupid'];
		$entity_group_order = $thisgroup['entitygrouporder'];
		$entity_group_collapsed = $thisgroup['entitygroupcollapse'];
		$entity_group_hidden = $thisgroup['entitygrouphidden'];
		$entity_state = 0; // start group

		eval('$groupbit = "' . fetch_template('links_addnewcat_oneentity') . '";');

		$entitybit = array();
		foreach ($entity_definitions['entities'] as $thisid=>$thisentity) {
			if ($thisentity['entitygroup']!=$entity_group) {
				continue;
			}
			$entity_type = $thisentity['entitytype'];
			if ($entity_type==ENTITY_TYPE_HIDDEN and !$showhidden) {
				continue;
			}
			$entity_name = $thisentity['entityname'];

			$entity_translated = "";
			if ($entity_name[0]=="*") {
				$entity_phrase = substr($entity_name,1,strlen($entity_name)-1);
				if (isset($vbphrase[$entity_phrase])) {
					$entity_translated = $vbphrase[$entity_phrase];
				}
			}

			$entity_display = $thisentity['displayorder'];
			$entity_required = $thisentity['required'];
			$entity_state = 1; // show entity
			$entity_seq = $thisid;
			$entity_count = $thisentity['entitycount'];
			eval('$entitybit[$entity_display] .= "' . fetch_template('links_addnewcat_oneentity') . '";');
		}

		if (count($entitybit)) {
			ksort($entitybit);
			$entitybits .= $groupbit . implode('', $entitybit);
		}

	}

	switch ($is_override) {
	case 1:
		$entity_state = 2; // show revert option
		eval('$entitybits .= "' . fetch_template('links_addnewcat_oneentity') . '";');
		break;
	case 2:
		$entity_state = 3; // show inherited
		eval('$entitybits .= "' . fetch_template('links_addnewcat_oneentity') . '";');
		break;
	}

	return $entitybits;

}

/**
* Set up the 'Entry Attributes' bits for selected linkids
*
* @param	intarr	Array of linkids
* @param	str		Display template for entitybit
* @param	str		Display template for type-related markup within entitybit
* @param	bool	Show hidden entities
* @return	strarr	Array of constructed entitybits
*/

function ldm_get_entitybits($linkids, $template, $markuptemplate, $showhidden=0, $dohighlight=0, $highlight_find="", $highlight_repl="") {
	global $vbulletin, $stylevar, $vbphrase;
	global $LINKS_SCRIPT, $RESIZE_SCRIPT, $SEARCH_SCRIPT;
	global $BASE_CAT;
	global $links_defaults, $links_permissions, $LDM_environment;
	global $ldm_icon_cache;

	$linkentityarray = array();
	$linkid = 0;

	$ENTITY_TYPE_YESNO = ENTITY_TYPE_YESNO;
	$ENTITY_TYPE_TEXT = ENTITY_TYPE_TEXT;
	$ENTITY_TYPE_URL = ENTITY_TYPE_URL;
	$ENTITY_TYPE_MEDIA = ENTITY_TYPE_MEDIA;
	$ENTITY_TYPE_UPLOAD = ENTITY_TYPE_UPLOAD;
	$ENTITY_TYPE_MEDIAUPLOAD = ENTITY_TYPE_MEDIAUPLOAD;
	$ENTITY_TYPE_IMAGE = ENTITY_TYPE_IMAGE;
	$ENTITY_TYPE_IMAGEUPLOAD = ENTITY_TYPE_IMAGEUPLOAD;
	$ENTITY_TYPE_HIDDEN = ENTITY_TYPE_HIDDEN;

	$linksavetext = (is_browser('mozilla') ? $vbphrase['ll_visitdownload_moz'] : $vbphrase['ll_visitdownload']);
	eval("\$entitydownload .= \"".fetch_template('links_downloadbit')."\";");

	if (count($linkids)) {

		$entity_bycat_definitions = array();
		foreach ($linkids as $linkid=>$linkcatid) {
			if (!isset($entity_bycat_definitions[$linkcatid])) {
				$entity_bycat_definitions[$linkcatid] = ldm_lookup_composite_entitydef($linkcatid);
			}
		}

		$query = "
			SELECT links.linkid AS linkid, links.linkurl AS linkurl, entities.entityid AS entityid,
			entities.entityname AS entityname, entities.entityvalue AS entityvalue,
			entities.entitygroup AS entitygroup, entities.entitytype AS entitytype,
			entities.entitydesc AS entitydesc, entities.entitydate AS entitydate
			FROM ". THIS_TABLE . "linksentities AS entities
			LEFT JOIN ". THIS_TABLE . "linkslink AS links
			ON entities.linkid=links.linkid
			WHERE links.linkid IN (".implode(',', array_keys($linkids)).")
			";

		$entity_values = array();
		$asb = $vbulletin->db->query_read($query);
		while ($myrow = $vbulletin->db->fetch_array($asb)) {
			$entity_values[$myrow['entityid']] = $myrow;
		}
		$vbulletin->db->free_result($asb);

		foreach ($entity_values as $thisentityid=>$thisentity) {
			$linkcatid = $linkids[$thisentity['linkid']];
			foreach ($entity_bycat_definitions[$linkcatid]['entities'] as $thisentitydef) {
				if ($thisentitydef['entityname']==$thisentity['entityname']) {
					$entity_values[$thisentityid]['displayorder'] = $thisentitydef['displayorder'];
					break;
				}
			}
		}
		uasort($entity_values, 'ldm_entitybit_sort');

		foreach ($linkids as $linkid=>$linkcatid) {

			$open_bit = 0;

			($hook = vBulletinHook::fetch_hook('ldm_getentitybit_start')) ? eval($hook) : false;

			foreach ($entity_bycat_definitions[$linkcatid]['groups'] as $thisgroup) {

				if ($thisgroup['entitygrouphidden']) {
					continue;
				}

				if ($thisgroup['entitygrouporder']<=0 and !$links_permissions['can_view_hidden']) {
					continue;
				}

				$open_group = 0;

				foreach ($entity_values as $thisentity) {

					if ($thisentity['linkid']!=$linkid) {
						continue;
					}

					$entitygroup = $thisentity['entitygroup'];
					if ($entitygroup!=$thisgroup['entitygroup']) {
						continue;
					}

					$entitygroupname = $entitygroup;
					if ($entitygroupname[0]=="*") {
						$entityphrase = substr($entitygroupname,1,strlen($entitygroupname)-1);
						if (isset($vbphrase[$entityphrase])) {
							$entitygroupname = $vbphrase[$entityphrase];
						}
					}

					$entityname = $thisentity['entityname'];
					if ($entityname[0]=="*") {
						$entityphrase = substr($entityname,1,strlen($entityname)-1);
						if (isset($vbphrase[$entityphrase])) {
							$entityname = $vbphrase[$entityphrase];
						}
					}

					$displayorder = $thisentity['displayorder'];
					if ($displayorder<0 and !$links_permissions['can_view_hidden']) {
						continue;
					}

					$entityid = $thisentity['entityid'];
					$entityvalue = $thisentity['entityvalue'];
					$entitydesc = ldm_parse_features(trim($thisentity['entitydesc']));
					$entitytype = $thisentity['entitytype'];
					$entitydate = $thisentity['entitydate'];
					$dateandtime = ldm_date($vbulletin->options['dateformat'], $thisentity['entitydate']).' '.ldm_date($vbulletin->options['timeformat'], $thisentity['entitydate']);
					$entitymarkup = "";

					if (!$open_bit) {
						$stage = 1;
						eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");
						$open_bit = 1;
						$open_group = 0;
						unset($last_display);
					}

					if (!$open_group) {
						$stage = 2;
						eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");
						$open_group = 1;
						unset($last_display);
						$shown_in_line = 0;
					}

					$entitymarkup = $entityicon = $entitysave = $entitysavetext = $entityreltag = $entitytitle = '';
					$entityisimage = $is_musicbox = $entitysearch = 0;

					switch ($entitytype) { // Build the type-dependent markup in the entitybit

					case ENTITY_TYPE_URL :
					case ENTITY_TYPE_UPLOAD :
						$urlInfo = ldm_parse_url($entityvalue);
						$urlType = file_extension($urlInfo['path']);
						$entity_known_mimetype = ldm_known_filetype($urlType);
						$entitysavetext = ($entity_known_mimetype ? $linksavetext : $vbphrase['ll_visiturl']);
						$entityurl = ldm_seo_url("jump", $linkcatid, $linkid, $entityid, $pagenumber);
						if ($entity_known_mimetype) {
							$entitysave = $entityurl;
						}
						$entityicon = ldm_get_typebit($urlInfo);
						$entitytemplate = 1;
    					if ($dohighlight and strlen($entitydesc)) {
	    					$entitydesc = ldm_apply_highlight($entitydesc, $highlight_find, $highlight_repl);
		    			}
						break;

					case ENTITY_TYPE_TEXT :
						$entitytext = ldm_parse_features(trim($entityvalue));
						if ($dohighlight) {
							$entitytext = ldm_apply_highlight($entitytext, $highlight_find, $highlight_repl);
        					if (strlen($entitydesc)) {
    	    					$entitydesc = ldm_apply_highlight($entitydesc, $highlight_find, $highlight_repl);
	    	    			}
						}
                        if ($links_permissions['can_search_link']) {
                            $entitysearch = 1;
                        }
						$entitytemplate = 4;
						break;

					case ENTITY_TYPE_HIDDEN :
						if ($links_permissions['can_view_hidden']) {
							$entitytext = ldm_parse_features(trim($entityvalue));
    						if ($dohighlight) {
	    						$entitytext = ldm_apply_highlight($entitytext, $highlight_find, $highlight_repl);
            					if (strlen($entitydesc)) {
    	        					$entitydesc = ldm_apply_highlight($entitydesc, $highlight_find, $highlight_repl);
	    	        			}
					    	}
							$entitytemplate = 4;
						}
						else {
							$entitytemplate = -1;
						}
						break;

					case ENTITY_TYPE_YESNO :
						$entitytemplate = 5;
						if ($dohighlight and strlen($entitydesc)) {
	    					$entitydesc = ldm_apply_highlight($entitydesc, $highlight_find, $highlight_repl);
						}
						break;

					case ENTITY_TYPE_MEDIA :
					case ENTITY_TYPE_MEDIAUPLOAD :
						$urlInfo = ldm_parse_url($entityvalue);
						$urlType = file_extension($urlInfo['path']);
						$entityicon = ldm_get_typebit($urlInfo);
						$entitymarkup = $entityname;
						if ($links_permissions['can_play_musicbox']) {
							$entityurl = ldm_seo_url("play", $linkcatid, $linkid, $entityid, $pagenumber);
							$entitysave = ldm_seo_url("jump", $linkcatid, $linkid, $entityid, $pagenumber);
							$entitywindow = ($links_defaults['open_musicbox_newwindow'] ? 'target="player" onclick="ldm_popup(this.href);return false;"' : 'target="_top"');
							$entitytitle = $vbphrase[ll_playme];
							$is_musicbox = 1;
						}
						if ($dohighlight and strlen($entitydesc)) {
	    					$entitydesc = ldm_apply_highlight($entitydesc, $highlight_find, $highlight_repl);
						}
						$entitytemplate = 6;
						break;

					case ENTITY_TYPE_IMAGE :
					case ENTITY_TYPE_IMAGEUPLOAD :
						if ($links_defaults['link_imagesize']) {
							$entityimgurl = $RESIZE_SCRIPT.'.php?'.$vbulletin->session->vars['sessionurl'].'linkid='.$linkid.'&amp;entityid='.$entityid;
							$entitysave = ldm_seo_url("jump", $linkcatid, $linkid, $entityid, $pagenumber);
							$linkimg = 1; // to tell the template to deliver
							eval("\$entityimgmag = \"".fetch_template('links_imgmag')."\";");
							$entityimg = ldm_get_imagebit($entityimgurl.'&amp;size='.$links_defaults['entity_imagesize']);
							$entitytemplate = 2;
							$entityisimage = 1;
						}
						if ($dohighlight and strlen($entitydesc)) {
	    					$entitydesc = ldm_apply_highlight($entitydesc, $highlight_find, $highlight_repl);
						}
						break;
					}

					($hook = vBulletinHook::fetch_hook('ldm_getentitybit_oneitem')) ? eval($hook) : false;

					eval("\$entitymarkup = \"".fetch_template($markuptemplate)."\";");

					if (isset($last_display)) {
						if ($displayorder != $last_display) {
							$stage = 3;
							eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");
							$shown_in_line = 0;
						}
						elseif ($shown_in_line >= $links_defaults['entity_perrow']) {
							$stage = 3;
							eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");
							$shown_in_line = 0;
						}
					}
					else {
						$shown_in_line = 0;
					}

					$last_display = $displayorder;
					$shown_in_line += 1;

					$stage = 4;
					eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");

				}

				if ($open_group) {
					$stage = 5;
					eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");
				}

			}

			if ($open_bit) {
				$stage = 6;
				eval("\$linkentityarray[$linkid] .= \"".fetch_template($template)."\";");
			}

		}

	}

	($hook = vBulletinHook::fetch_hook('ldm_getentitybits_complete')) ? eval($hook) : false;

	return $linkentityarray;

}

/**
* Check consistencies between the entities table and the corresponding admin settings
* in *entity_definitions* and update/correct *entity_definitions* accordingly
*
* @param	&arr	Entity records
* @param	&arr	Admin entity definition array, updated in situ
* @return	void
*/

function ldm_check_entity_consistency(&$entities, &$entity_definitions) {

// Step 0 : zero all the counts
	foreach ($entity_definitions['entities'] as $thisentityid=>$thisentity) {
		$entity_definitions['entities'][$thisentityid]['entitycount'] = 0;
	}
	foreach ($entity_definitions['groups'] as $thisgroupid=>$thisgroup) {
		$entity_definitions['groups'][$thisgroupid]['entitycount'] = 0;
	}

// Step 1 : check that all groups defined in $entity_definitions['entities'] also appear in ['groups']
	foreach ($entity_definitions['entities'] as $thisentityid=>$thisentity) {
		$entity_definitions['entities'][$thisentityid]['entitycount'] = 0;
		$entity_group = $thisentity['entitygroup'];
		$found = 0;
		foreach ($entity_definitions['groups'] as $thisgroup) {
			if ($thisgroup['entitygroup']==$entity_group) {
				$found = 1;
				break;
			}
		}

		if (!$found) {
			$nextkey = 0;
			if (count($entity_definitions['groups'])) {
				$nextkey = max(0,max(array_keys($entity_definitions['groups']))+1);
			}
			$entity_definitions['groups'][$nextkey] = array(
				'entitygroupid' => $nextkey,
				'entitygroup' => $entity_group,
				'entitygrouporder' => ($entity_group==ENTITY_ADMIN_GROUP ? -1 : 1),
				'entitycount' => 0,
				);
		}
	}

// Step 2 : check that all entities/groups in $entities also appear in $entity_definitions['entities']/['groups']
	foreach ($entities as $thisentityname=>$thisentity) {
		$found = 0;
		foreach ($entity_definitions['entities'] as $thisentitydefid=>$thisentitydef) {
			if ($thisentityname==$thisentitydef['entityname']) {
				$found = 1;
				break;
			}
		}

		if ($found) {
			continue; // entity definition already matches
		}

		$found = 0;
		foreach ($entity_definitions['groups'] as $thisgroup) {
			if ($thisgroup['entitygroup']==$thisentity['entitygroup']) {
				$found = 1;
				break;
			}
		}

		if (!$found) {
			$nextkey = 0;
			if (count($entity_definitions['groups'])) {
				$nextkey = max(0, max(array_keys($entity_definitions['groups']))+1);
			}
			$entity_definitions['groups'][$nextkey] = array(
				'entitygroupid' => $nextkey,
				'entitygroup' => $thisentity['entitygroup'],
				'entitygrouporder' => ($thisentity['entitygroup']==ENTITY_ADMIN_GROUP ? -1 : 1),
				'entitycount' => 0,
				);
		}
		$found = 0;

		foreach ($entity_definitions['entities'] as $thisentitydef) {
			if ($thisentitydef['entityname']==$thisentityname) {
				$found = 1;
				break;
			}
		}

		if (!$found) {
			$entity_definitions['entities'][] = array(
				'entityname' => $thisentityname,
				'entitygroup' => $thisentity['entitygroup'],
				'entitytype' => $thisentity['entitytype'],
				'entitycount' => 0,
				'displayorder' => 1,
				);
		}
	}

// Step 3 : correct the counts
	foreach ($entities as $thisentityname=>$thisentity) {
		foreach ($entity_definitions['entities'] as $thisentitydefid=>$thisentitydef) {
			if ($thisentityname==$thisentitydef['entityname']) {
				$entity_definitions['entities'][$thisentitydefid]['entitycount'] = $thisentity['entitycount'];
				break;
			}
		}
	}
	foreach ($entity_definitions['entities'] as $thisentitydefid=>$thisentitydef) {
		foreach ($entity_definitions['groups'] as $thisgroupdefid=>$thisgroupdef) {
			if ($thisgroupdef['entitygroup']==$thisentitydef['entitygroup']) {
				$entity_definitions['groups'][$thisgroupdefid]['entitycount'] += 1;
				break;
			}
		}
	}

// Step 4 : get rid of empty groups
	foreach ($entity_definitions['groups'] as $thisgroupid=>$thisgroup) {
		if ($thisgroup['entitycount']==0) {
			unset($entity_definitions['groups'][$thisgroupid]);
		}
	}

}

/**
* Force consistencies between the entities table and the corresponding admin settings
* in *entity_definitions* by correcting and updating tables where necessary
*
*/

function ldm_force_entity_consistency() {
	global $vbulletin;

// Read the entity data table
	$query = "
		SELECT entityname, COUNT(linkid) AS entitycount, entitygroup, entitytype
		FROM ". THIS_TABLE . "linksentities
		GROUP BY entityname
		";
	$asb = $vbulletin->db->query_read($query);
	$entities = array();
	while ($rec=$vbulletin->db->fetch_array($asb)) {
		$entities[$rec['entityname']]['entitygroup'] = $rec['entitygroup'];
		$entities[$rec['entityname']]['entitycount'] = $rec['entitycount'];
		$entities[$rec['entityname']]['entitytype'] = $rec['entitytype'];
	}

// Fetch what's currently in the database, as an addin might have changed this
	$query = "
		SELECT setting
		FROM ".THIS_TABLE."linksadmin
		WHERE settingname='entity_definitions'
		AND catid=-1
		";
	$row = $vbulletin->db->query_first($query);
	$entity_definitions = unserialize($row['setting']);

// Check consistency
	ldm_check_entity_consistency($entities, $entity_definitions);

// Update the data dictionary
	$query = "
		UPDATE ".THIS_TABLE."linksadmin
		SET
			setting='".$vbulletin->db->escape_string(serialize($entity_definitions))."'
		WHERE settingname='entity_definitions'
		AND catid=-1
		";
	$vbulletin->db->query_write($query);

	ldm_datastore_markdirty('ldm_admin');

// Update the entries
	foreach ($entity_definitions['entities'] as $thisentity) {
		$query = "
			UPDATE ". THIS_TABLE . "linksentities
			SET
				entitygroup = '".$vbulletin->db->escape_string($thisentity['entitygroup'])."',
				entitytype = '".intval($thisentity['entitytype'])."'
			WHERE entityname = '".$vbulletin->db->escape_string($thisentity['entityname'])."'
			";
		$vbulletin->db->query_write($query);
	}

}

?>