<?php
/**
 * @package JFusion
 * @subpackage Models
 * @author JFusion development team
 * @copyright Copyright (C) 2008 JFusion. All rights reserved.
 * @license http://www.gnu.org/copyleft/gpl.html GNU/GPL
 */

// Check to ensure this file is included in Joomla!
defined('_JEXEC') or die();
 /**
 * Require the Joomla Installer model
 */
require_once(JPATH_ADMINISTRATOR .DS.'components'.DS.'com_installer'.DS.'models'.DS.'install.php');

/**
 * Class to manage plugin install in JFusion
 * @package JFusion
 */
class JFusionModelInstaller extends InstallerModelInstall {

	/** @var object JTable object */
	var $_table = null;

	/** @var object JTable object */
	var $_url = null;

	/**
	 * Overridden constructor
	 * @access	protected
	 */
	function __construct()
	{
		// Load the language file
		$lang =& JFactory::getLanguage();
		$lang->load('com_installer');
		parent::__construct();

	}

	/**
	 * Replaces original Install() method.
	 * @return true|false Result of the JFusion plugin install
	 */
	function install() {
		global $mainframe;

		$this->setState('action', 'install');

		switch(JRequest::getWord('installtype'))
		{
			case 'folder':
				$package = $this->_getPackageFromFolder();
				break;

			case 'upload':
				$package = $this->_getPackageFromUpload();
				break;

			case 'url':
				$package = $this->_getPackageFromUrl();
				break;

			default:
				$this->setState('message', JText::_('NO_INSTALL_TYPE'));
				$result = false;
                return $result;
				break;
		}

		// Was the package unpacked?
		if (!$package) {
			$this->setState('message', JText::_('NO_PACKAGE_FOUND') );
            $result = false;
            return $result;
		}

		// custom installer
		$installer = new JfusionPluginInstaller($this);
		// Install the package
		if (!$installer->install($package['dir'])) {
			// There was an error installing the package
			$msg = 'JFusion ' . JText::_('PLUGIN') . ' ' . JText::_('INSTALL') . ' ' . JText::_('FAILED');
			$result = false;
		} else {
			// Package installed sucessfully
			$msg = 'JFusion ' . JText::_('PLUGIN') . ' ' . JText::_('INSTALL') . ' ' . JText::_('SUCCESS');
			$result = true;
		}
		// Set some model state values
		$mainframe->enqueueMessage($msg);
		$this->setState('name', $installer->get('name'));
		$this->setState('result', $result);
		$this->setState('message', $installer->parent->message);
		$this->setState('extension.message', $installer->get('extension.message'));

		// Cleanup the install files
		if (!is_file($package['packagefile'])) {
			$config =& JFactory::getConfig();
			$package['packagefile'] = $config->getValue('config.tmp_path').DS.$package['packagefile'];
		}

		JInstallerHelper::cleanupInstall($package['packagefile'], $package['extractdir']);

		return $result;


	}

/**
 * Installer class for JFusion plugins
 * @param string $jname name of the JFusion plugin used
 */
	function uninstall($jname) {
		$db =& JFactory::getDBO();
		$db->setQuery('SELECT id FROM #__jfusion WHERE name ='. $db->Quote($jname));
		$myId = $db->loadResult();
		if (!$myId) {
			;// error!! plugin not installed (hack attempt?)
		}

		$installer = new JfusionPluginInstaller($this);
		// Install the package
		if (!$installer->uninstall($jname)) {
			// There was an error installing the package
			$msg = 'JFusion ' . JText::_('PLUGIN') . ' ' . JText::_('UNINSTALL') . ' ' . JText::_('FAILED');
			$result = false;
		} else {
			// Package installed sucessfully
			$msg = 'JFusion ' . JText::_('PLUGIN') . ' ' . JText::_('UNINSTALL') . ' ' . JText::_('SUCCESS');
			$result = true;
		}

		return $result;

	}

	function copy($jname, $new_jname, $update = false) {
		$db =& JFactory::getDBO();
		$db->setQuery('SELECT id FROM #__jfusion WHERE name ='. $db->Quote($jname));
		$myId = $db->loadResult();
		if (!$myId) {
			;// error!! plugin not installed (hack attempt?)
		}

		$installer = new JfusionPluginInstaller($this);
		// Install the package
		if (!$installer->copy($jname, $new_jname, $update)) {
			// There was an error installing the package
			$msg = 'JFusion ' . JText::_('PLUGIN') . ' ' . JText::_('COPY') . ' ' . JText::_('FAILED');
			$result = false;
		} else {
			// Package installed sucessfully
			$msg = 'JFusion ' . JText::_('PLUGIN') . ' ' . JText::_('COPY') . ' ' . JText::_('SUCCESS');
			$result = true;
		}

		return $result;
	}
}

/**
 * Installer class for JFusion plugins
 * @package JFusion
 */
class JFusionPluginInstaller extends JObject {

	function __construct(&$parent) {

		$this->parent =& JInstaller::getInstance();
		$this->parent->setOverwrite(true);
	}

	/**
	 * handles JFusion plugin installation
     * @param string $dir install path
	 * @return boolean
	 */
	function install($dir=null) {
		// Get a database connector object
		$db =& JFactory::getDBO();

		if ($dir && JFolder::exists($dir)) {
			$this->parent->setPath('source', $dir);

		} else {
			$this->parent->abort(JText::_('INSTALL_INVALID_PATH'));
            $result = false;
            return $result;
		}

		// Get the extension manifest object
		$manifest = $this->_getManifest($dir);
		if(is_null($manifest)) {
			$this->parent->abort(JText::_('INSTALL_NOT_VALID_PLUGIN'));
            $result = false;
            return $result;
		}

		$this->manifest =& $manifest->document;
		//var_dump($this->manifest);

		/**
		 * ---------------------------------------------------------------------------------------------
		 * Manifest Document Setup Section
		 * ---------------------------------------------------------------------------------------------
		 */

		// Set the extensions name
		$name =& $this->manifest->getElementByPath('name');
		$name = JFilterInput::clean($name->data(), 'string');
		$this->set('name', $name);


		// Get the component description
		$description = & $this->manifest->getElementByPath('description');
		if (is_a($description, 'JSimpleXMLElement')) {
			$this->parent->set('message', $description->data());
		} else {
			$this->parent->set('message', '' );
		}
		$myDesc = JFilterInput::clean($description->data(), 'string');

		// installation path
		$this->parent->setPath('extension_root', JPATH_ADMINISTRATOR.DS.'components'.DS.'com_jfusion'.DS.'plugins'.DS. $name);

		// get files to copy
		$element =& $this->manifest->getElementByPath('files');

		/**
		 * ---------------------------------------------------------------------------------------------
		 * Filesystem Processing Section
		 * ---------------------------------------------------------------------------------------------
		 */
		// If the plugin directory does not exist, lets create it
		$created = false;
		if (!file_exists($this->parent->getPath('extension_root'))) {
			if (!$created = JFolder::create($this->parent->getPath('extension_root'))) {
				$this->parent->abort(JText::_('PLUGIN').' '.JText::_('INSTALL').': '.JText::_('INSTALL_FAILED_DIRECTORY').': "'.$this->parent->getPath('extension_root').'"');
            	$result = false;
            	return $result;
			}
		}

		/*
		 * If we created the plugin directory and will want to remove it if we
		 * have to roll back the installation, lets add it to the installation
		 * step stack
		 */
		if ($created) {
			$this->parent->pushStep(array ('type' => 'folder', 'path' => $this->parent->getPath('extension_root')));
		}

		// Copy all necessary files
		if ($this->parent->parseFiles($element, -1) === false) {
			// Install failed, roll back changes
			$this->parent->abort();
            $result = false;
            return $result;
		}

		/**
		 * ---------------------------------------------------------------------------------------------
		 * Language files Processing Section
		 * ---------------------------------------------------------------------------------------------
		 */
		$this->parent->parseLanguages($this->manifest->getElementByPath('languages'));
		$this->parent->parseLanguages($this->manifest->getElementByPath('administration/languages'), 1);
		
		/**
		 * ---------------------------------------------------------------------------------------------
		 * Database Processing Section
		 * ---------------------------------------------------------------------------------------------
		 */
		//determine the features of the plugin
		$features = array('master','slave','dual_login','check_encryption','activity','search','discussion');
		foreach($features as $f) {
			$xml = $this->manifest->getElementByPath($f);
			if (is_a($xml, 'JSimpleXMLElement')) {
				$$f = JFilterInput::clean($xml->data(), 'integer');
			} elseif ($f=='master' && $f=='check_encryption') {
				$$f = 0;
			} else {
				$$f = 3;
			}
		}

		//let's check to see if a plugin with the same name is already installed
		$db->setQuery('SELECT id, '.implode(', ',$features).' FROM #__jfusion WHERE name = '.$db->Quote($name));
		$plugin = $db->loadObject();

		if (!empty($plugin)) {
			if (!$this->parent->getOverwrite()) {
				// Install failed, roll back changes
				$this->parent->abort(JText::_('PLUGIN').' '.JText::_('Install').': '.JText::_('PLUGIN').' "'.$name.'" '.JText::_('ALREADY_EXISTS'));
	            $result = false;
    	        return $result;
			} else {
				//enable/disable features and update the plugin files
				
				//store enabled/disabled features to update copies
				global $plugin_features;
				$plugin_features = array();
				$plugin_files = $this->backup($name);				
				$query = 'UPDATE #__jfusion SET plugin_files = '.$db->Quote($plugin_files);
				foreach($features as $f) {
					if(($$f==3 && $plugin->$f!=3) || ($$f!=3 && $plugin->$f==3)){
						$query .= ', '.$f.'='.$$f;
						$plugin_features[$f] = $$f;
					}
				}
				$query .= " WHERE id = ".$plugin->id;
				$db->setQuery($query);
				$db->query();
			}
		} else {
            //prepare the variables
            $plugin_entry = new stdClass;
            $plugin_entry->id = NULL;
            $plugin_entry->name = $name;
			$plugin_entry->dual_login = $dual_login;
			$plugin_entry->slave = $slave;
			$plugin_entry->activity = $activity;
			$plugin_entry->search = $search;
			$plugin_entry->discussion = $discussion;
			$plugin_entry->plugin_files = $this->backup($name);

            //now append the new plugin data
			if (!$db->insertObject('#__jfusion', $plugin_entry, 'id' )){
		        // Install failed, roll back changes
		        $this->parent->abort(JText::_('PLUGIN').' '.JText::_('INSTALL').' '.JText::_('ERROR').': ' . $db->stderr());
		        return false;
			}
			$this->parent->pushStep(array ('type' => 'plugin', 'id' => $plugin_entry->id));
		}

		/**
		 * ---------------------------------------------------------------------------------------------
		 * Finalization and Cleanup Section
		 * ---------------------------------------------------------------------------------------------
		 */

		// Lastly, we will copy the manifest file to its appropriate place.
		if (!$this->parent->copyManifest(-1)) {
			// Install failed, rollback changes
			$this->parent->abort(JText::_('PLUGIN').' '.JText::_('INSTALL').': '.JText::_('INSTALL_ERROR_FILE'));
            $result = false;
            return $result;
		}

		//check to see if this is updating a plugin that has been copied
		$query = "SELECT name FROM #__jfusion WHERE original_name = {$db->Quote($name)}";
		$db->setQuery($query);
		$copiedPlugins = $db->loadObjectList();

		foreach($copiedPlugins as $plugin){
			//update the copied version with the new files
			$this->copy($name,$plugin->name,true);
		}

        $result = true;
        return $result;
	}

	/**
	 * handles JFusion plugin un-installation
     * @param string $jname name of the JFusion plugin used
	 * @return boolean
	 */
	function uninstall($jname) {
		$dir = JPATH_ADMINISTRATOR.DS.'components'.DS.'com_jfusion'.DS.'plugins'.DS. $jname;

		if (!$jname || !JFolder::exists($dir)) {
			$this->parent->abort(JText::_('UNINSTALL_ERROR_PATH'));
            $result = false;
            return $result;
		}
		
		/**
		 * ---------------------------------------------------------------------------------------------
		 * Remove Language files Processing Section
		 * ---------------------------------------------------------------------------------------------
		 */
		// Get the extension manifest object
		$manifest = $this->_getManifest($dir);
		if(is_null($manifest)) {
			$this->parent->abort(JText::_('INSTALL_NOT_VALID_PLUGIN'));
            $result = false;
            return $result;
		}

		$this->manifest =& $manifest->document;
		$this->parent->removeFiles($this->manifest->getElementByPath('languages'));
		$this->parent->removeFiles($this->manifest->getElementByPath('administration/languages'), 1);
		
		// remove files
		if (!JFolder::delete($dir)) {
			$this->parent->abort(JText::_('UNINSTALL_ERROR_DELETE'));
            $result = false;
            return $result;
		}
		
		$db =& JFactory::getDBO();

		// delete raw
		$db->setQuery('DELETE FROM #__jfusion WHERE name = '. $db->Quote($jname));
		if(!$db->query()) {
			$this->parent->abort(JText::_('Owned!'));
		}

		$db->setQuery('DELETE FROM #__jfusion_user_plugin WHERE jname = ' . $db->Quote($jname));
		if(!$db->query()) {
			$this->parent->abort(JText::_('Owned!'));
		}
		
		$db->setQuery('DELETE FROM #__jfusion_forum_plugin WHERE jname = ' . $db->Quote($jname));
		if(!$db->query()) {
			$this->parent->abort(JText::_('Owned!'));
		}		
	}

	/**
	 * handles copying JFusion plugins
     * @param string $jname name of the JFusion plugin used
     * @param string $new_jname name of the copied plugin
     * @param boolean $update mark if we updating a copied plugin
	 * @return boolean
	 */
	function copy($jname, $new_jname, $update = false) {

		$dir = JPATH_ADMINISTRATOR.DS.'components'.DS.'com_jfusion'.DS.'plugins'.DS. $jname;
		$new_dir = JPATH_ADMINISTRATOR.DS.'components'.DS.'com_jfusion'.DS.'plugins'.DS. $new_jname;

		if (!$jname || !JFolder::exists($dir)) {
			$this->parent->abort(JText::_('COPY_ERROR_PATH'));
            $result = false;
            return $result;
		}

		//copy the files
		if (!JFolder::copy($dir, $new_dir, null, $update)) {
			$this->parent->abort(JText::_('COPY_ERROR'));
            $result = false;
            return $result;
		}

		// Define our preg arrays
		$regex		= array();
		$replace	= array();

		//change the classname
		$regex[]	= '#JFusion(Auth|User|Forum|Public|Admin)_' . $jname .'#ms';
		$replace[]	= 'JFusion$1_' . $new_jname;

		//change the jname function
		$regex[]	= '#return \'' . $jname .'\';#ms';
		$replace[]	= 'return \'' . $new_jname .'\';';

		//update the XML name tag
		$regex[]	= '#<name>' . $jname .'</name>#ms';
		$replace[]	= '<name>' . $new_jname .'</name>';
		
		/**
		 * ---------------------------------------------------------------------------------------------
		 * Copy Language files Processing Section
		 * @todo - This section may be improved but works actually
		 * ---------------------------------------------------------------------------------------------
		 */
		$manifest = $this->_getManifest($dir);
		if(is_null($manifest)) {
			$this->parent->abort(JText::_('INSTALL_NOT_VALID_PLUGIN'));
            $result = false;
            return $result;
		}

		$this->manifest =& $manifest->document;
		
		$childrens = array();
		$path = "";
		$languages = & $this->manifest->getElementByPath ( 'languages' );
		$adminLanguages = & $this->manifest->getElementByPath ( 'administration/languages' );
			
		// Copy of site languages files
		if (is_a ( $languages, 'JSimpleXMLElement' )) $childrens = $languages->children ();
		if (count ( $childrens ) > 0) {
			foreach ( $childrens as $children ) {
				$attributes = $children->_attributes;
				
				$filepath = $children->_data;
				// If the language file as a path in the content of the element language
				if(strpos( $filepath, DS)){
					$folders = explode( DS ,$filepath);
					array_pop($folders); // remove the file name of the path
					$path = implode(DS , $folders) . DS;
				}
				// @todo - take in consideration if the filename doesn't respect xx-XX.com_jfusion.plg_pluginname.ini
				$prefix = $attributes ['tag'] . '.com_jfusion.plg_';
				$languageFile = $prefix . $jname . '.ini';
				$newLanguageFile = $prefix . $new_jname . '.ini';
				if (file_exists ( JPATH_SITE . DS . "language" . DS . $attributes ['tag'] . DS . $languageFile )) {
					if (! JFile::copy ( JPATH_SITE . DS . "language" . DS . $attributes ['tag'] . DS . $languageFile, JPATH_SITE . DS . "language" . DS . $attributes ['tag'] . DS . $newLanguageFile )) {
						$this->parent->abort ( JText::_ ( 'COPY_ERROR' ) );
						$result = false;
						return $result;
					}
				}
				// @todo - Need to be improved because it includes admin and site languages files. But at the end it's supposed to be the same name normally!
				$regex [] = '#<language tag="' . $attributes ['tag'] . '">' . $path . $languageFile . '</language>#ms';
				$replace [] = '<language tag="' . $attributes ['tag'] . '">' . $path . $newLanguageFile . '</language>';
			}
		}
		// Copy of admin languages files		
		if (is_a ( $adminLanguages, 'JSimpleXMLElement' )) $childrens = $adminLanguages->children ();
		if (count ( $childrens ) > 0) {
			foreach ( $childrens as $children ) {
				$attributes = $children->_attributes;
				
				$filepath = $children->_data;
				// If the language file as a path in the content of the element language
				if(strpos( $filepath, DS)){
					$folders = explode( DS ,$filepath);
					array_pop($folders); // remove the file name of the path
					$path = implode(DS, $folders) . DS;
				}
				// @todo - take in consideration if the filename doesn't respect xx-XX.com_jfusion.plg_pluginname.ini	
				$prefix = $attributes ['tag'] . '.com_jfusion.plg_';
				$languageFile = $prefix . $jname . '.ini';
				$newLanguageFile = $prefix . $new_jname . '.ini';
				if (file_exists ( JPATH_ADMINISTRATOR . DS . "language". DS . $attributes ['tag'] . DS . $languageFile )) {
					if(!JFile::copy ( JPATH_ADMINISTRATOR . DS . "language" . DS . $attributes ['tag'] . DS . $languageFile, JPATH_ADMINISTRATOR . DS . "language" . DS . $attributes ['tag'] . DS . $newLanguageFile )){
						$this->parent->abort ( JText::_ ( 'COPY_ERROR' ) );
						$result = false;
						return $result;
					}
				}
				// @todo - Need to be improved because it includes admin and site languages files. But at the end it's supposed to be the same name normally!
				$regex [] = '#<language tag="' . $attributes ['tag'] . '">' . $path . $languageFile . '</language>#ms';
				$replace [] = '<language tag="' . $attributes ['tag'] . '">' . $path . $newLanguageFile . '</language>';
			}
		}
		
		/**
		 * ---------------------------------------------------------------------------------------------
		 * Rename class files and xml file of the new plugin create
		 * ---------------------------------------------------------------------------------------------
		 */
		
		//define which files need parsing
		$parse_files = array($new_dir . DS . 'auth.php', $new_dir . DS . 'admin.php', $new_dir . DS . 'user.php', $new_dir . DS . 'jfusion.xml', $new_dir . DS . 'forum.php', $new_dir . DS . 'public.php');

		foreach ($parse_files as $parse_file) {
			if(file_exists($parse_file)){
				$file_data = JFile::read($parse_file);
				$file_data = preg_replace($regex, $replace, $file_data);
        		JFile::write($parse_file, $file_data);
			}
		}

		$db =& JFactory::getDBO();
		if($update) {
			//update the copied plugin files
			$plugin_files = $this->backup($new_jname);
			$query = "UPDATE #__jfusion SET plugin_files = ".$db->Quote($plugin_files);
			
			//get the features of the updated plugin
			global $plugin_features;
			
			if(empty($plugin_features)) {
				//copy() was called directly because we are upgrading the component
				$features = array('master','slave','dual_login','check_encryption','activity','search','discussion');
				foreach($features as $f) {
					$xml = $this->manifest->getElementByPath($f);
					if (is_a($xml, 'JSimpleXMLElement')) {
						$$f = JFilterInput::clean($xml->data(), 'integer');
					} elseif ($f=='master' && $f=='check_encryption') {
						$$f = 0;
					} else {
						$$f = 3;
					}
				}

				$db->setQuery('SELECT id '.implode(', ',$features).' FROM #__jfusion WHERE name = '.$db->Quote($new_jname));
				$plugin = $db->loadObject();

				if (!empty($plugin)) {
					//enable/disable features and update the plugin files
					$plugin_features = array();
					foreach($features as $f) {
						if(($$f==3 && $plugin->$f!=3) || ($$f!=3 && $plugin->$f==3)){
							$plugin_features[$f] = $$f;
						}
					}
				} else {
					$plugin_features = array();
				}
			}
			
			foreach($plugin_features as $key => $val) {
				$query .= ", $key = $val";	
			}
			
			$query .= " WHERE name = '$new_jname'";
			$db->setQuery($query);
			$db->query();
			
		} else {
			//add the new entry in the JFusion plugin table
			$db->setQuery('SELECT * FROM #__jfusion WHERE name = '.$db->Quote($jname));
			$plugin_entry = $db->loadObject();
			$plugin_entry->name = $new_jname;
			$plugin_entry->id = NULL;
			$plugin_entry->master = ($plugin_entry->master == 3) ? 3 : 0;
			$plugin_entry->slave = ($plugin_entry->slave == 3) ? 3 : 0;
			$plugin_entry->plugin_files = $this->backup($new_jname);

			//only change the original name if this is not a copy itself
			if(empty($plugin_entry->original_name)) {
				$plugin_entry->original_name = $jname;
			}

	        if (!$db->insertObject('#__jfusion', $plugin_entry, 'id' )) {
	            //return the error
	            $this->parent->abort('Error while creating the plugin: ' . $db->stderr());
	        }
		}

        $result = true;
        return $result;
	}


	/**
	 * load manifest file with installation information
	 * @return simpleXML object (or null)
	 * @param $dir string - Directory
	 */
	function _getManifest($dir) {
		// Initialize variables
		$null	= null;
		$xml	=& JFactory::getXMLParser('Simple');

		// TODO: DISCUSS if we should allow flexible naming for installation file
		$this->parent->setPath('manifest', $dir. DS .'jfusion.xml');
		// If we cannot load the xml file return null
		if (!$xml->loadFile($dir.DS.'jfusion.xml')) {
			// Free up xml parser memory and return null
			unset ($xml);
			return $null;
		}

		/*
		 * Check for a valid XML root tag.
		 * @todo: Remove backwards compatability in a future version
		 * Should be 'install', but for backward compatability we will accept 'mosinstall'.
		 */
		$root =& $xml->document;
		if (!is_object($root) || ($root->name() != 'install' && $root->name() != 'mosinstall')) {
			// Free up xml parser memory and return null
			unset ($xml);
			return $null;
		}

		// Valid manifest file return the object
		return $xml;
	}

	/**
	 * handles JFusion plugin backups
     * @param string $jname name of the JFusion plugin used
	 * @return backup zip file data or location
	 */
	function backup($jname)
	{
		$config =& JFactory::getConfig();
		$tmpDir =& $config->getValue('config.tmp_path');

		//compress the files
		$filename = $tmpDir.DS.$jname.'.zip';
		//retrieve a list of files within the plugin directory
		$pluginPath = JPATH_ADMINISTRATOR .DS.'components'.DS.'com_jfusion'.DS.'plugins'.DS.$jname;
		//check for zip creation
		$zipSuccess = false;
		//we need to chdir into the plugin path
		$cwd = getcwd();
		chdir($pluginPath);	
		//get the contents of the files in the plugin dir	
		$filesArray = $this->getFiles($pluginPath,$jname);

		if(extension_loaded('zlib')) {
			//use Joomla's zip class to create the zip
	        $zip =& JArchive::getAdapter('zip');
	        if($zip->create($filename, $filesArray)) {
	        	$zipSuccess = true;
	        }
	        
		} elseif(class_exists('ZipArchive')) {
			//use PECL ZipArchive to create the zip
			$zip = new ZipArchive();
			if($zip->open($filename, ZIPARCHIVE::CREATE)===TRUE) {
				foreach($filesArray as $file) {
					$zip->addFromString($file['name'],$file['data']);
				}
				$zip->close();
				$zipSuccess = true;
			}
		}
		
		chdir($cwd);
		$data = ($zipSuccess && file_exists($filename)) ? @file_get_contents($filename) : '';
		
		return $data;
	}
	
	function getFiles($folder,$jname) 
	{
		$filesArray = array();
		$files = JFolder::files($folder,null,false,true);
		foreach($files as $file)
        {
        	$file = str_replace(JPATH_ADMINISTRATOR .DS.'components'.DS.'com_jfusion'.DS.'plugins'.DS.$jname.DS,'',$file);
            $data = JFile::read($file);
            $filesArray[] = array('name' => $file, 'data' => $data);
        }
        $folders = JFolder::folders($folder,null,false,true);
        if(!empty($folders)){
        	foreach($folders as $f) {
        		$filesArray = array_merge($filesArray,$this->getFiles($f,$jname));
        	}
        }
        
        return $filesArray;
	}
}