<?php
/**
 * aLinks WordPress Plugin
 *
 * A WordPress plugin that automatically links keywords in your
 * blog post.
 *
 * @author    Sean Hickey <seanhickey@gmail.com>
 * @package    alinks
 * @license    GPL
 * @version    $Id: aLinksLogger.php 44 2008-03-24 18:00:53Z sean $
 */
class aLinksLogger {

	/**
	 * Full path to the log file directory.
	 * @var string
	 */
	var $logDir;
	
	/**
	 * Column to order return results by.
	 * @var string
	 */
	var $orderBy;
	
	/**
	 * Either 'ASC' or 'DESC' to order SELECT
	 * results ascending or descending.
	 * @var string
	 */
	var $sortOrder;
	
	var $html;
	
	/**
	 * Class constructor
	 *
	 * @param	string	$logDir	The full path to the log files directory
	 * @return	void
	 */
	function aLinksLogger($logDir) {
		$this->logDir = ALINKS_LOG_PATH;
	}
	
	/**
	 * Proccess a query on log files and return the results of that query
	 *
	 * This is the main method for the class.  Most other methods are called directly
	 * or indirectly by this method.  Takes a query string based on SQL and performs
	 * the query on log file(s).  Returns either an array of results from the query,
	 * or a message string if something is wrong with the query.
	 *
	 * @param	string	$query	The query to perform on the log file(s)
	 * @return	mixed
	 */
	function query($query, $buildHtml = false) {

		$queries = explode(';', $query);
		//print_r($queries);
		
		$returnVals = array();
		foreach ($queries as $query) {
			$query = trim($query);
			if (empty($query)) continue;
			
			$tokens = explode(' ', $query, 2);
			switch (strtolower(trim($tokens[0]))) {
				case 'insert':
					$returnVals[] = $this->insert($query, $buildHtml);
					break;
				case 'delete':
					$returnVals[] = $this->delete($query, $buildHtml);
					break;
				case 'select':
					$returnVals[] = $this->select($query, $buildHtml);
					break;
				case 'show':
					$returnVals[] = $this->show($query, $buildHtml);
					break;
				case 'create':
					$returnVals[] = $this->create($query, $buildHtml);
					break;
			}
		}
		
		if (count($returnVals) == 1) {
			$returnVals = $returnVals[0];
		}
		return $returnVals;
	}
	
	/**
	 * Creates a log file
	 *
	 * Used when the query to perform uses the CREATE TABLE syntax.  Returns either
	 * message string if something is wrong with the query, or 1 if the log file
	 * was successfully created.
	 *
	 * @param	string	$query	The full query to perform
	 * @return	mixed
	 */
	function create($query, $buildHtml = false) {

		if (!preg_match('~CREATE TABLE [\\\'|"|`](.*?)[\\\'|"|`]~is', $query, $matches)) {
			return '<strong>Query Error:</strong> Unknown syntax';
		}
		
		$logFile = $this->trimQuotes($matches[1]);
		if (!is_writable($this->logDir)) {
			return '<strong>Query Error: Log directory is not writable';
		}
		
		file_put_contents($this->logDir . '/' . $logFile . '.log', '');
		
		if ($buildHtml) {
			$this->html .= "<h3>CREATE TABLE $logFile</h3>\r\n";
			$this->html .= "<p>Table $logFile created</p>";
		}
		return 1;
	}
	
	/**
	 * Inserts a record or records into a log file
	 *
	 * Used when the INSERT INTO syntax is used.  The query can either accept values
	 * to insert using the VALUES() syntax, or a SELECT statement can be used to
	 * get records from another log file to insert into this log file.  Returns the
	 * number of records inserted into the log file, or a message string if something
	 * is wrong with the query.
	 *
	 * @param	string	$query	The full query to perform
	 * @return	mixed
	 */
	function insert($query, $buildHtml = false) {
		if (!preg_match('~INSERT INTO (.*)? ((VALUES\((.*)?,(.*)?,(.*)?\))|(SELECT.*))~is', $query, $matches)) {
			return '<strong>Query Error:</strong> Unknown syntax';
		}
		
		if (!empty($matches[7])) {
			$logFile = $matches[1];
			return;
			$insertData = $this->select($matches[7]);
		} else {
			$logFile = $this->trimQuotes($matches[1]);
			$timestamp = $this->trimQuotes($matches[5]);
			if (preg_match('~[a-zA-Z]~', $timestamp, $m)) {
				$timestamp = strtotime($timestamp);
			}
			$insertData = array(array('keyphrase' => $this->trimQuotes($matches[4]), 'timestamp' => $timestamp, 'ip' => $this->trimQuotes($matches[6])));
		}

		if (empty($logFile)) {
			return '<strong>Query Error:</strong> Log file cannot be null';
		}
		if (!file_exists($this->logDir . '/' . $logFile . '.log')) {
			return "<strong>Query Error:</strong> Log file `$logFile` does not exists";
		}

		$counter = 0;
		foreach ($insertData as $data) {
			$this->writeToLog($this->logDir . '/' . $logFile . '.log', $data['keyphrase'], $data['timestamp'], $data['ip']);
			$counter++;
		}
		
		if ($buildHtml) {
			$this->html .= "<h3>INSERT INTO $logFile</h3>\r\n";
			$this->html .= "<p>$counter records inserted</p>";
		}
		return $counter;
	}
	
	/**
	 * Deletes records from a log file
	 *
	 * The same as MySQL's DELETE statement, this method will delete records from
	 * the specified log file based on critieria specified in the query string.  Returns
	 * the number of records deleted, or a message string if something is wrong
	 * with the query.
	 *
	 * @param	string	$query	The query to perform on the log file
	 * @return	mixed
	 */
	function delete($query, $buildHtml = false) {
		if (!preg_match('~DELETE FROM `?(\w*)`?( WHERE `?(keyphrase|timestamp|ip)`? ([=<>\!]+) [\\\'|"](.*?)[\\\'|"])?~is', $query, $matches)) {
			return '<strong>Query Error:</strong> Unknown syntax';
		}

		$logFile = 	$this->trimQuotes($matches[1]);
		$where = 	$this->trimQuotes($matches[3]);
		$operand = 	$this->trimQuotes($matches[4]);
		$value = 	$this->trimQuotes($matches[5]);
		
		if (strcasecmp($where, 'timestamp') === 0) {
			if (preg_match('~[a-zA-Z]~', $value, $matches)) {
				$value = strtotime($value);
			}
		}

		if ($operand == '=') {
			$operand = '==';
		}
		if ($operand == '=>') {
			$operand = '>=';
		}
		if ($operand == '=<') {
			$operand = '<=';
		}
		
		if (empty($logFile)) {
			return '<strong>Query Error:</strong> Log file cannot be null';
		}
		if (!file_exists($this->logDir . '/' . $logFile . '.log')) {
			return "<strong>Query Error:</strong> Log file `$logFile` does not exists";
		}
		
		if (!is_writable($this->logDir)) {
			return "<strong>Query Error:</strong> Log file directoy is not writable.";
		}
		
		$fR = @fopen($this->logDir . '/' . $logFile . '.log', 'rb');
		if (!$fR) {
			return '<strong>Query Error:</strong> Error opening log file';
		}

		$founds = array();
		$counter = 0;
		$deleted = 0;
		while ($record = $this->readFromLog($fR, $counter)) {
		
			if (!empty($where)) {
				$code = "
				if (\$record[\$where] $operand \$value) {
					\$deleted++;
				} else {
					\$founds[] = \$record;
				}
				";
				eval($code);
			}
			$counter++;
		}
		
		@fclose($fR);
		$fR = @fopen($this->logDir . '/' . $logFile . '.log', 'wb');
		if (!$fR) {
			return '<strong>Query Error:</strong> Error opening log file';
		}
		
		foreach ($founds as $found) {
			$this->writeToLog($fR, $found['keyphrase'], $found['timestamp'], $found['ip']);
		}
		@fclose($fR);
		
		if ($buildHtml) {
			$this->html .= "<h3>DELETE FROM $logFile</h3>\r\n";
			$this->html .= "<p>$deleted records deleted</p>";
		}
		return $deleted;
	}
	
	/**
	 * Returns an array of records from a log file
	 *
	 * Pretty much the same as MySQL's SELECT statement, this method performs a SELECT
	 * query on a log file, and returns the records that match the criteria specified
	 * in the query.  Returns either an array of results, or a message string if something
	 * is wrong with the query.
	 *
	 * @param	string	$query	The query to perform
	 * @return	mixed
	 */
	function select($query, $buildHtml = false) {

		if (!preg_match('~SELECT (DISTINCT)?(.*?) FROM `?(\w*)`?( WHERE `?(keyphrase|timestamp|ip)`? ([=<>\!]+) [\\\'|"](.*?)[\\\'|"])?( AND `?(keyphrase|timestamp|ip)`? ([=<>\!]+) [\\\'|"](.*?)[\\\'|"])?( ORDER BY `?(keyphrase|timestamp|ip)`? (ASC|DESC))?~is', $query, $matches)) {
			return '<strong>Query Error:</strong> Unknown syntax';
		}

		$distinct = 	$this->trimQuotes($matches[1]);
		$columns =	$this->trimQuotes($matches[2]);
		$logFile = 	$this->trimQuotes($matches[3]);
		$where = 	$this->trimQuotes($matches[5]);
		$operand = 	$this->trimQuotes($matches[6]);
		$value = 	$this->trimQuotes($matches[7]);
		$andWhere =	$this->trimQuotes($matches[9]);
		$andOperand =	$this->trimQuotes($matches[10]);
		$andValue =	$this->trimQuotes($matches[11]);
		$orderBy =	$this->trimQuotes($matches[13]);
		$sortOrder =	$this->trimQuotes($matches[14]);
		
		if (strcasecmp($where, 'timestamp') === 0) {
			if (preg_match('~[a-zA-Z]~', $value, $matches)) {
				$value = strtotime($value);
			}
		}
		
		if (strcasecmp($andWhere, 'timestamp') === 0) {
			if (preg_match('~[a-zA-Z]~', $andValue, $matches)) {
				$andValue = strtotime($andValue);
			}
		}
		
		if ($operand == '=') {
			$operand = '==';
		}
		if ($operand == '=>') {
			$operand = '>=';
		}
		if ($operand == '=<') {
			$operand = '<=';
		}
		
		if ($andOperand == '=') {
			$andOperand = '==';
		}
		if ($andOperand == '=>') {
			$andOperand = '>=';
		}
		if ($andOperand == '=<') {
			$andOperand = '<=';
		}
		
		if (empty($logFile)) {
			return '<strong>Query Error:</strong> Log file cannot be null';
		}

		if (!file_exists($this->logDir . '/' . $logFile . '.log')) {
			return "<strong>Query Error:</strong> Log file `$logFile` does not exists";
		}
		
		$fR = @fopen($this->logDir . '/' . $logFile . '.log', 'rb');
		if (!$fR) {
			return '<strong>Query Error:</strong> Error opening log file';
		}
		
		$counter = 0;
		$founds = array();
		while ($record = $this->readFromLog($fR, $counter)) {
		
			if (!empty($where)) {
				if (!empty($andWhere)) {
					$code = "
					if (\$record[\$where] $operand \$value && \$record[\$andWhere] $andOperand \$andValue) {
						\$founds[] = \$record;
					}
					";
				} else {
					$code = "
					if (\$record[\$where] $operand \$value) {
						\$founds[] = \$record;
					}
					";
				}
				eval($code);
			} else {
				$founds[] = $record;
			}
			$counter++;
		}
		
		@fclose($fR);
		
		$toReturn = array();
		$counter = 0;
		foreach ($founds as $found) {
			
			if (stristr($columns, '*')) {
				$toReturn[$counter] = $found;
				$counter++;
				continue;
			}
			
			if (stristr($columns, 'keyphrase')) {
				$toReturn[$counter]['keyphrase'] = $found['keyphrase'];
			}
			if (stristr($columns, 'timestamp')) {
				$toReturn[$counter]['timestamp'] = $found['timestamp'];
			}
			if (stristr($columns, 'ip')) {
				$toReturn[$counter]['ip'] = $found['ip'];
			}
			$counter++;
		}
		
		if (!empty($orderBy)) {
			$toReturn = $this->sortRows($toReturn, $orderBy, $sortOrder);
		}
		
		if (!empty($distinct)) {
			$toReturn = $this->removeDup($toReturn);
			$nArray = array();
			$counter = 0;
			foreach ($toReturn as $tReturn) {
				$nArray[$counter] = $tReturn;
				$counter++;
			}
			$toReturn = $nArray;
		}
		
		if ($buildHtml) {
			$nToReturn = $this->ts2dt($toReturn, get_option('date_format') . ' ' . get_option('time_format'));
			$this->html .= "<h3>SELECT FROM $logFile</h3>\r\n";
			$this->html .= '<table width="100%">';
			$this->html .= "<tr>\r\n";
			if (isset($nToReturn[0]['keyphrase'])) {
				$this->html .= "<td><strong>Keyphrase</strong></td>\r\n";
			}
			if (isset($nToReturn[0]['timestamp'])) {
				$this->html .= "<td><strong>Timestamp</strong></td>\r\n";
			}
			if (isset($nToReturn[0]['ip'])) {
				$this->html .= "<td><strong>IP</strong</td>\r\n";
			}
				
			$this->html .= "</tr>\r\n";
			foreach ($nToReturn as $result) {
				$this->html .= "<tr>\r\n";
				if (isset($result['keyphrase'])) $this->html .= "<td>{$result['keyphrase']}</td>\r\n";
				if (isset($result['timestamp'])) $this->html .= "<td>{$result['timestamp']}</td>\r\n";
				if (isset($result['ip'])) $this->html .= "<td>{$result['ip']}</td>\r\n";
				$this->html .= "</tr>\r\n";
			}
			$this->html .= '</table>';
		}
		return $toReturn;
	}

	/**
	 * Returns a list of existing log files
	 *
	 * Used similar to MySQL's SHOW TABLES, this method will return a list of
	 * existing log files, or if the LIKE statement is used, will return
	 * the specified log file if it exists.  Returns a message string if something
	 * is wrong with the query.
	 *
	 * @param	string	$query	The query to perform
	 * @return	mixed
	 */
	function show($query, $buildHtml = false) {
		if (!preg_match('~SHOW TABLES( LIKE)?(.*)?~is', $query, $matches)) {
			return '<strong>Query Error:</strong> Unknown syntax';
		}
		
		$table = $matches[2];
		$dirs = $this->getDirs($this->logDir);
		
		if ($table) {
			$table = $this->trimQuotes($table);
			if (in_array($table, $dirs)) {
				return array($table);
			} else {
				return 0;
			}
		}
		
		if ($buildHtml) {
			$this->html .= "<h3>SHOW TABLES</h3>\r\n";
			foreach ($dirs as $dir) {
				$this->html .= "<p><strong>TABLE:</strong> $dir</p>";
			}
		}
		return $dirs;
	}
	
	/**
	 * Sorts return results from SELECT statement
	 *
	 * Takes an array of results and sorts the array based on the key
	 * specified by $orderBy, and sorts the array ascending or descending
	 * which is specified by $sortOrder.  Returns the sorted array.
	 *
	 * @param	array	$data		The array to sort
	 * @param	string	$orderBy		The key (column) to sort by
	 * @param	string	$sortOrder	Either 'ASC' or 'DESC'
	 * @return	array
	 */
	function sortRows($data, $orderBy, $sortOrder) {
		
		$this->orderBy = $orderBy;
		$this->sortOrder = $sortOrder;
		usort($data, array(&$this, 'sortCmp'));
		return $data;
	}
	
	/**
	 * Sort an array by key either ascending or descending
	 *
	 * Used by the usort() function in the sortRows() method.
	 *
	 * @return	int
	 */
	function sortCmp($a, $b) {
		$ord = strcasecmp($a[$this->orderBy], $b[$this->orderBy]);
		if (strtolower($this->sortOrder) == 'asc') {
			return $ord;
		} else {
			if ($ord < 0) {
				return 1;
			} else if ($ord > 0) {
				return -1;
			} else {
				return 0;
			}
		}
	}

	/**
	 * Remove duplicate entries from an array
	 *
	 * Used when DISTINCT is specified in a SELECT statement, this method returns
	 * an array will duplicate entries taken out.
	 *
	 * @param	array	$matriz	The array
	 * @return	array
	 */
	function removeDup($matriz) {
		$aux_ini = array();
		$entrega = array();
		for($n = 0; $n < count($matriz); $n++) {
			$aux_ini[] = serialize($matriz[$n]);
		}
		$mat = array_unique($aux_ini);
		for($n = 0; $n < count($matriz); $n++) {
			$entrega[] = unserialize($mat[$n]);
		}
		foreach ($entrega as $key => $row){
			if (!is_array($row)) { unset($entrega[$key]); }
		}
		return $entrega;
	}
	
	/**
	 * Gets a list of directories in a top directory
	 *
	 * Scans through the given directory and returns an array of
	 * all sub directories.  Returns FALSE on failure.
	 *
	 * @param	string	$dir	The name of the directory to scan
	 * @return	mixed
	 */
	function getDirs($dir) {
	    if(is_dir($dir)) {

			$dirlist = opendir($dir);
			while(($file = readdir($dirlist)) !== false) {
				if($file != '.' && $file != '..') {
					$dirs[] = str_replace('.log', '', $file);
				}
			}
	        return $dirs;
		} else {
			return false;
			break;
		}
	}
	
	/**
	 * Remove quotes and extra whitespace from a string
	 *
	 * Strips out any quote characters (`'") from a string, and trims whitespace
	 * from the ends of it.  Returns the stripped string.
	 *
	 * @param	string	$string	The string to strip
	 * @return	string
	 */
	function trimQuotes($string) {
		$string = str_replace('"', '', $string);
		$string = str_replace("'", '', $string);
		$string = str_replace('`', '', $string);
		return trim($string);
	}
	
	/**
	 * Appends a record to a log file
	 *
	 * Appends a record to the end of a log file.  Returns TRUE if the record
	 * was successfully added to the log file, or FALSE on failure.
	 *
	 * @param	string	$logFile		The full path to the log file to write to
	 * @param	string	$keyphrase	The keyphrase for the record
	 * @param	int		$timestamp	Unix timestamp for the record
	 * @param	string	$ipAddress	The IP address for the record
	 * @return	bool
	 */
	function writeToLog($logFile, $keyphrase, $timeStamp, $ipAddress) {
	
		if (is_resource($logFile)) {
			$fR = $logFile;
		} else {
			$dir = dirname($logFile);
			if (!is_writable($dir)) {
				return false;
			}
			$fR = fopen($logFile, 'ab');
		}
		if (strlen($keyphrase) > 120) {
			$keyphrase = substr($keyphrase, 0, 120);
		}
		
		$ips = explode('.', $ipAddress);
		$line = pack('A120L1C4', $keyphrase, $timeStamp, (int) $ips[0], (int) $ips[1], (int) $ips[2], (int) $ips[3]);
		fwrite($fR, $line);
		
		if (!is_resource($logFile)) {
			fclose($fR);
		}
		return true;
	}
	
	/**
	 * Returns a record from a log file
	 *
	 * Returns the specified record from an already opened log file.  Returns
	 * an array for the record, or FALSE if the EOF has been reached.
	 *
	 * @param	resource	$fR		The resource for the already opened log file
	 * @param	int		$recNum	The number of the record to return
	 * @return	array
	 */
	function readFromLog($fR, $recNum) {

		$bytes = $recNum * 128;

		$result = fseek($fR, $bytes);
		if ($result === -1) {
			return false;
		}
		$con = fread($fR, 128);
		if (empty($con)) {
			return false;
		}
		$log = unpack('A120keyphrase/L1timestamp/C4', $con);
		$ip = $log[1] . '.' . $log[2] . '.' . $log[3] . '.' . $log[4];
		unset($log[1], $log[2], $log[3], $log[4]);
		$log['ip'] = $ip;
		return $log;
	}
	
	/**
	 * Converts timestamps in array to a date format
	 *
	 * Takes all the timestamp elements in an array, who's values should be Unix
	 * timestamps, and converts them to a data format specified by $dataFormat.
	 *
	 * @param	array	$data		The array to convert
	 * @param	string	$dateFormat	The format for the date() function
	 * @return	array
	 */
	function ts2dt($data, $dateFormat) {
		
		for ($i = 0; $i < count($data); $i++) {
			if (!isset($data[$i]['timestamp'])) continue;
			$data[$i]['timestamp'] = date($dateFormat, $data[$i]['timestamp']);
		}
		return $data;
	}
}