<?php
/**
 * aLinks WordPress Plugin
 *
 * A WordPress plugin that automatically links keywords in your
 * blog post.
 *
 * @author    Sean Hickey <seanhickey@gmail.com>
 * @package   alinks
 * @license   GPL
 * @version   $Id: aLinksImportExport.php 44 2008-03-24 18:00:53Z sean $
 */

 /**
  * aLinks import export class
  *
  * Used for importing and exporting aLinks keyphrases, settings, and
  * modules.
  */
class aLinksImportExport extends aLinksApp {

	/**
	 * The WPDB object.
	 * @var obj
	 */
	var $wpdb;
	
	/**
	 * The string prefix for all tables in WordPress.
	 * @var string
	 */
	var $table_prefix;
	
	/**
	 * Class constructor
	 *
	 * @param	obj		$wpdb		The WPDB object
	 * @param	string	$table_prefix	The string prefix for all tables in WordPress
	 * @return	void
	 */
	function aLinksImportExport(&$wpdb, $table_prefix) {
		$this->wpdb = &$wpdb;
		$this->table_prefix = $table_prefix;
	}

	/**
	 * Exports aLinks keyphrases, settings, and modules
	 *
	 * Exports, to a file, the user created keyphrases.  If there use selects to have
	 * the general settings exported, those will also be included in the file.  If the user
	 * selectes to have the modules exported, those will also be included in the file.
	 * The export file is broken up into sections, and each section starts with a "header"
	 * to describe the section.  The headers are:
	 *
	 * #%%KEYPHRASES#***#
	 * #%%#SETTINGS#**#
	 * #%%#SETTINGS_{mod_setting_name}#**# where {mod_setting_name} is the setting name for a module
	 * #%%#MODULE_{archive}#**# where {archive} is the name of the ZIP archive in the export file
	 *
	 * The headers use the special #%%##**# format to ensure a keyphrase, or any other data in
	 * the file isn't confused with a header.  They need to be fairly unique.
	 * The keyphrases following the keyphrase header are the 7 column values of each keyphrase row
	 * in the database seperated by a newline `\n`.
	 * The settings following the settings header is each setting on a newline with the name of
	 * the setting seperated by the value of the setting with a colon `:`. Ex foo:bar
	 * The settings for each module following the settings_mod_settings header is just the serialized
	 * values for that module.
	 * The module data following the module header is a string of ZIP compressed content for that
	 * module.
	 *
	 * All of the data is base 64 encoded to help identify corrupot files, and to keep users from
	 * messing around with the internals of the file, which could mess things up.
	 * The file starts with a MD5 hash of the base 64 data, that will be used to make sure
	 * none of the data was currupted.  The hash is followed by a newline, and then the base 64
	 * data.
	 *
	 * A typical export file, *not* base 64 encoded, looks like this:
	 *
	 * #%%KEYPHRASES#***#
	 * book
	 * Get books at amazon.com
	 * a:2:{s:10:"searchterm";s:12:"harry potter";s:11:"producttype";s:5:"books";}
	 * amazon basic
	 * 2006-05-08 11:34:56
	 * adf4sdfbn43089adfgjasdfklj430ugdf
	 * #%%#SETTINGS#**#
	 * linkimage:http://www.mysite.com/link.gif
	 * targetwindow:_blank
	 * clicktracking:on
	 * customhtmljs:
	 * cssclass:alinks_link
	 * #%%#SETTINGS_alinks_mod_amazon#**#
	 * a:4:{s:12:"setting_name";s:17:"alinks_mod_amazon";s:8:"amazonid";s:11:"myassoc-id";s:9:"amazonurl";s:12:"amazon.co.jp";}
	 * #%%#MODULE_amazon#**#
	 * {Binary data here for a ZIP file}
	 *
	 * The whole thing is converted into base 64, a MD5 hash is made for the base 64 string, and the file is sent to
	 * user looking *something* like this:
	 *
	 * asdf94tnsdfbvkire90u4kmdfkmrd934arfdsgkl
	 * sddfofdgidfkklaskwejfroidfsgoit595pij4ksadfkfdvkdfidfsgirisdafkljsdf
	 * alsadfuierijfkjdflkdfsgijdfgoijdrgf9uerpodfgkdfskjdfgsijdfirpuifpdfs
	 *
	 * Where the first line is the MD5 hash, then a new line, and then the base 64 data
	 *
	 * @param	array	$post	The $_POST data from the export form
	 * @return	void
	 */
	function export($post) {
		$exportAll = $post['exportall'];
		if (empty($exportAll)) {
			$exportAll = 'all';
		}
		
		$lastExport = get_option('alinks_last_export');
		if (empty($lastExport)) {
			$exportAll = 'all';
		}

		if (isset($post['exportsettings'])) {
			$exportSettings = get_option('alinks_settings');
		}
		if (isset($post['exportmodules'])) {
			$archives = $this->getModules();
		}
		
		if ($exportAll == 'all') {
			$keyphrases = $this->wpdb->get_results("SELECT * FROM `{$this->table_prefix}alinks_keyphrases`", ARRAY_A);
		} else {
			$keyphrases = $this->wpdb->get_results("SELECT * FROM `{$this->table_prefix}alinks_keyphrases` WHERE `created` > '$lastExport'", ARRAY_A);
		}
		
		$exportString = "#%%#KEYPHRASES#**#\n";
		foreach ($keyphrases as $keyphrase) {
			$exportString .= $keyphrase['keyphrase'] . "\n";
			$exportString .= $keyphrase['description'] . "\n";
			$exportString .= $keyphrase['defines'] . "\n";
			$exportString .= $keyphrase['module'] . "\n";
			$exportString .= $keyphrase['created'] . "\n";
			$exportString .= $keyphrase['hash'] . "\n";
		}
		
		if (isset($exportSettings)) {
			$exportString .= "#%%#SETTINGS#**#\n";
			$exportString .= "linkimage:{$exportSettings['linkimage']}\n";
			$exportString .= "newwindow:{$exportSettings['newwindow']}\n";
			$exportString .= "clicktracking:{$exportSettings['clicktracking']}\n";
			$exportString .= "customhtmljs:{$exportSettings['customhtmljs']}\n";
			$exportString .= "cssclass:{$exportSettings['cssclass']}\n";
			$exportString .= "maxlinks:{$exportSettings['maxlinks']}\n";
			$exportString .= "casesensitive:{$exportSettings['casesensitive']}\n";
			
			$modSettings = $this->wpdb->get_results	("
											SELECT option_name, option_value 
											FROM {$this->wpdb->options} 
											WHERE option_name LIKE '%alinks_mod_%'
												", ARRAY_A);
			if (!empty($modSettings)) {
				foreach ($modSettings as $modSetting) {
					$exportString .= "#%%#SETTINGS_{$modSetting['option_name']}#**#\n";
					$exportString .= $modSetting['option_value'] . "\n";
				}
			}
		}
		
		if (isset($archives)) {
			foreach ($archives as $archive) {
				$exportString .= "#%%#MODULE_{$archive}#**#\n";
				$exportString .= file_get_contents(ALINKS_TMP_PATH . '/' . $archive . '.zip') . "\n";
			}
		}
		
		$encodedString = base64_encode($exportString);
		$stringHash = md5($encodedString);
		
		update_option('alinks_last_export', date('Y-m-d H:i:s'));
		
		header("Content-type: application/alinkssp");
		header('Content-Disposition: attachment; filename="keyphrases-' . date('Y-m-d-H-i-s') . '.ale"');
		echo $stringHash . "\n";
		echo $encodedString;
		die();
	}
	
	/**
	 * Imports a ale file
	 *
	 * aLinks exports 'ale' files, and this method imports the data from one of those
	 * files.  It decodes the base 64 data in the file, checks the MD5 hash of that decoded
	 * data against the MD5 hash stored in the file to ensure the file isn't corrupt, and
	 * then breaks the file into each "part".  It them sends each part to a different
	 * method in this class to import that data for that part.
	 * Returns FALSE if there was no file uploaded, or if there was an error importing
	 * the file.  Returns TRUE on success.
	 *
	 * @param	array	$files	The $_FILES data created when a file is uploaded via a web form
	 * @param	array	$post	The $_POST data from the import form
	 * @return	bool
	 */
	function import($files, $post) {
		if (!isset($files['importfile']['tmp_name'])) {
			return false;
		}

		$contents = file_get_contents($files['importfile']['tmp_name']);
		$eContents = explode("\n", $contents, 2);
		
		$importHash = $eContents[0];
		$hashMatch = md5($eContents[1]);
		
		if (strcmp($importHash, $hashMatch) !== 0) {
			$this->addSessionMessage('Imported file appears to be corrupted.  Keyphrases not imported.', true);
			return false;
		}
		$contents = base64_decode($eContents[1]);
		$parts = explode('#%%#', $contents);

		for ($i = 0; $i < count($parts); $i++) {
			if (empty($parts[$i])) continue;
			if (strstr($parts[$i], 'KEYPHRASES')) {
				$this->importKeyphrases($parts[$i]);
			} else if (strstr($parts[$i], 'SETTINGS_') && isset($post['importsettings'])) {
				$this->importModSettings($parts[$i]);
			} else if (strstr($parts[$i], 'SETTINGS') && isset($post['importsettings'])) {
				$this->importGeneralSettings($parts[$i ]);
			} else if (strstr($parts[$i], 'MODULE_') && isset($post['importmodules'])) {
				$this->importModule($parts[$i]);
			}
		}
		
		$this->addSessionMessage('Keyphrase imported');
		return true;
	}
	
	/**
	 * Imports keyphrases in an ale file
	 *
	 * The import() method takes the keyphrases section of the ale file and passes it to
	 * this method.  This method then imports the keyphrases.
	 * Returns FALSE if there was an error importing the keyphrases, or TRUE on
	 * success.
	 *
	 * @param	string	$data	The keyphrases 'part' from the ale file
	 * @return	bool
	 */
	function importKeyphrases($data) {

		$sData = explode("\n", $data, 2);
		$sData = trim($sData[1]);
		$keyphrases = explode("\n", $sData);

		for ($i = 0; $i < count($keyphrases); $i += 6) {
			$phrase = $this->wpdb->escape($keyphrases[$i]);
			$description = $this->wpdb->escape($keyphrases[$i + 1]);
			$defines = $this->wpdb->escape($keyphrases[$i + 2]);
			$module = $this->wpdb->escape($keyphrases[$i + 3]);
			$created = $keyphrases[$i + 4];
			$hash = $keyphrases[$i + 5];
			
			$result = $this->wpdb->get_var("SELECT id FROM `{$this->table_prefix}alinks_keyphrases` WHERE `hash` = '$hash'");
			if (!$result) {
				$this->wpdb->query	("
								INSERT INTO `{$this->table_prefix}alinks_keyphrases` 
								VALUES (
								null, 
								'$phrase', 
								'$description', 
								'$defines', 
								'$module', 
								null, 
								'$hash'
								)
									");
			} else {
				$this->wpdb->query	("
								UPDATE `{$this->table_prefix}alinks_keyphrases` 
								SET `keyphrase` = '$phrase', 
								`description` = '$description', 
								`defines` = '$defines' 
								WHERE `hash` = '$hash'
									");
			}
		}
		return true;
	}

	/**
	 * Imports general settings in an ale file
	 *
	 * The import() method takes the settings section of the ale file and passes it to
	 * this method.  This method then imports the general settings.
	 * Returns FALSE if there was an error importing the settings, or TRUE on
	 * success.
	 *
	 * @param	string	$data	The settings 'part' from the ale file
	 * @return	bool
	 */
	function importGeneralSettings($data) {
		$sData = explode("\n", $data, 2);
		$sData = trim($sData[1]);
		
		$settings = explode("\n", $sData);
		$nSettings = array();
		foreach ($settings as $setting) {
			$eSettings = explode(':', $setting, 2);
			$nSettings[$eSettings[0]] = $eSettings[1];
		}
		update_option('alinks_settings', $nSettings);
		return true;
	}
	
	/**
	 * Imports module settings in an ale file
	 *
	 * The import() method takes the module settings section of the ale file and passes it to
	 * this method.  This method then imports the module settings.
	 * Returns FALSE if there was an error importing the module settings, or TRUE on
	 * success.
	 *
	 * @param	string	$data	The module settings 'part' from the ale file
	 * @return	bool
	 */
	function importModSettings($data) {
		$sData = explode("\n", $data, 2);
		$name = str_replace('SETTINGS_', '', $sData[0]);
		$name = str_replace('#**#', '', $name);
		$sData = trim($sData[1]);
		$eData = unserialize($sData);
		update_option($name, $eData);	
	}
	
	/**
	 * Imports module ZIP files from the ale file
	 *
	 * Each module that the user has installed when they export an ale file is compressed
	 * into a ZIP file, and then that file's contents are added to the ale file.  This
	 * method takes that content and writes it back out to a ZIP file, and then uncompresses
	 * the ZIP file to the modules directory.
	 * Returns FALSE if there was an error, or TRUE on success.
	 *
	 * @param	string	$data	The modules 'part' from the ale file
	 * @return	bool
	 */
	function importModule($data) {
		require_once(ALINKS_CLASSES_PATH . '/pclzip.lib.php');
		$sData = explode("\n", $data, 2);
		$dirName = str_replace('MODULE_', '', $sData[0]);
		$dirName = str_replace('#**#', '', $dirName);
		$sData = $sData[1];
		if (!is_writable(ALINKS_TMP_PATH)) {
			$this->addSessionMessage('Temp directory not writable.  Modules not imported.', true);
			return false;
		}
		file_put_contents(ALINKS_TMP_PATH . "/$dirName", $sData);
		$archive = new PclZip(ALINKS_TMP_PATH . "/$dirName");
		$archive->extract(PCLZIP_OPT_PATH, ALINKS_MODULES_PATH);
		return true;
	}
	
	/**
	 * Creates ZIP files for all installed modules and returns a list of those ZIP files
	 *
	 * Scans the modules directory for subdirectories, which will be modules.  Each of those
	 * sub directories is ZIPed up and stored in the temp directory.  An array of each
	 * module directory that was zipped up is returned, otherwise FALSE is returned if there
	 * was an error.
	 *
	 * @return	mixed
	 */
	function getModules() {
		require_once(ALINKS_CLASSES_PATH . '/pclzip.lib.php');
		$dirs = $this->getDirs(ALINKS_MODULES_PATH);
		if (!$dirs) {
			return false;
		}
		
		if (!is_writable(ALINKS_TMP_PATH)) {
			return false;
		}
		
		$archives = array();
		foreach ($dirs as $dir) {
			$archive = new PclZip(ALINKS_TMP_PATH . '/' . $dir . '.zip');
			$fList = $archive->create(ALINKS_MODULES_PATH . '/' . $dir, PCLZIP_OPT_REMOVE_ALL_PATH );
			if ($fList != 0) {
				$archives[] = $dir;
			}
		}
		return $archives;
	}
}