<?php
/**
 */
# Copyright (C) 2004 Brion Vibber <brion@pobox.com>
# http://www.mediawiki.org/
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
# http://www.gnu.org/copyleft/gpl.html

/**
 * Authentication plugin interface. Instantiate a subclass of AuthPlugin
 * and set $wgAuth to it to authenticate against some external tool.
 *
 * The default behavior is not to do anything, and use the local user
 * database for all authentication. A subclass can require that all
 * accounts authenticate externally, or use it only as a fallback; also
 * you can transparently create internal wiki accounts the first time
 * someone logs in who can be authenticated externally.
 */

require_once($IP."/includes/AuthPlugin.php");

class vBSSO extends AuthPlugin {
	/**
	 * Check whether there exists a user account with the given name.
	 * The name will be normalized to MediaWiki's requirements, so
	 * you might need to munge it (for instance, for lowercase initial
	 * letters).
	 *
	 * @param $username String: username.
	 * @return bool
	 * @public
	 */
	function userExists( $username ) {
		# Override this!
		return false;
	}

	/**
	 * Check if a username+password pair is a valid login.
	 * The name will be normalized to MediaWiki's requirements, so
	 * you might need to munge it (for instance, for lowercase initial
	 * letters).
	 *
	 * @param $username String: username.
	 * @param $password String: user password.
	 * @return bool
	 * @public
	 */
	function authenticate( $username, $password ) {
		# Override this!
		return false;
	}

	/**
	 * Modify options in the login template.
	 *
	 * @param $template UserLoginTemplate object.
	 * @public
	 */
	function modifyUITemplate( &$template ) {
		# Override this!
		$template->set( 'usedomain', false );
	}

	/**
	 * Set the domain this plugin is supposed to use when authenticating.
	 *
	 * @param $domain String: authentication domain.
	 * @public
	 */
	function setDomain( $domain ) {
		$this->domain = $domain;
	}

	/**
	 * Check to see if the specific domain is a valid domain.
	 *
	 * @param $domain String: authentication domain.
	 * @return bool
	 * @public
	 */
	function validDomain( $domain ) {
		# Override this!
		return true;
	}

	/**
	 * When a user logs in, optionally fill in preferences and such.
	 * For instance, you might pull the email address or real name from the
	 * external user database.
	 *
	 * The User object is passed by reference so it can be modified; don't
	 * forget the & on your function declaration.
	 *
	 * @param User $user
	 * @public
	 */
	function updateUser( &$user ) {
		# Override this and do something
		return true;
	}


	/**
	 * Return true if the wiki should create a new local account automatically
	 * when asked to login a user who doesn't exist locally but does in the
	 * external auth database.
	 *
	 * If you don't automatically create accounts, you must still create
	 * accounts in some way. It's not possible to authenticate without
	 * a local account.
	 *
	 * This is just a question, and shouldn't perform any actions.
	 *
	 * @return bool
	 * @public
	 */
	function autoCreate() {
		return false;
	}

	/**
	 * Can users change their passwords?
	 *
	 * @return bool
	 */
	function allowPasswordChange() {
		return true;
	}

	/**
	 * Set the given password in the authentication database.
	 * As a special case, the password may be set to null to request
	 * locking the password to an unusable value, with the expectation
	 * that it will be set later through a mail reset or other method.
	 *
	 * Return true if successful.
	 *
	 * @param $user User object.
	 * @param $password String: password.
	 * @return bool
	 * @public
	 */
	function setPassword( $user, $password ) {
		return true;
	}

	/**
	 * Update user information in the external authentication database.
	 * Return true if successful.
	 *
	 * @param $user User object.
	 * @return bool
	 * @public
	 */
	function updateExternalDB( $user ) {
		return true;
	}

	/**
	 * Check to see if external accounts can be created.
	 * Return true if external accounts can be created.
	 * @return bool
	 * @public
	 */
	function canCreateAccounts() {
		return false;
	}

	/**
	 * Add a user to the external authentication database.
	 * Return true if successful.
	 *
	 * @param User $user - only the name should be assumed valid at this point
	 * @param string $password
	 * @param string $email
	 * @param string $realname
	 * @return bool
	 * @public
	 */
	function addUser( $user, $password, $email='', $realname='' ) {
		return true;
	}


	/**
	 * Return true to prevent logins that don't authenticate here from being
	 * checked against the local database's password fields.
	 *
	 * This is just a question, and shouldn't perform any actions.
	 *
	 * @return bool
	 * @public
	 */
	function strict() {
		return false;
	}

	/**
	 * Check if a user should authenticate locally if the global authentication fails.
	 * If either this or strict() returns true, local authentication is not used.
	 *
	 * @param $username String: username.
	 * @return bool
	 * @public
	 */
	function strictUserAuth( $username ) {
		return false;
	}

	/**
	 * When creating a user account, optionally fill in preferences and such.
	 * For instance, you might pull the email address or real name from the
	 * external user database.
	 *
	 * The User object is passed by reference so it can be modified; don't
	 * forget the & on your function declaration.
	 *
	 * @param $user User object.
	 * @param $autocreate bool True if user is being autocreated on login
	 * @public
	 */
	function initUser( $user, $autocreate=false ) {
		# Override this to do something.
	}

	/**
	 * If you want to munge the case of an account name before the final
	 * check, now is your chance.
	 */
	function getCanonicalName( $username ) {
		return $username;
	}
}

# INITIALIZE VBULLETIN SUBSYSTEM
if ( !defined( 'CWD' ) )
    define( 'CWD', ( VB_SYSTEM_PATH ) );

define('THIS_SCRIPT', 'wiki');
require_once( CWD . '/global.php');
require_once( CWD . '/includes/functions.php');

$vBDonor = FALSE;
if ( is_member_of( $vbulletin->userinfo, 11 ) ) {
    $vBDonor = TRUE;
}

# END VBULLETIN SUBSYSTEM

function vBSSOAutoAuth(&$user) {
        global $wgRequest;
 	global $vbulletin;
	global $wgUser;

        if ( $user != null ) {
                // user is authenticated (by another hook)
		if ( !$user->isAnon() ) {
			// User is not anonymous. 
			// Check for the existence of a valid vB userid.  If we don't have one, log the old user out.

			if ( $vbulletin->userinfo['userid'] == 0 ) {
				$user->logout();
			}

			// Check NoLogin groups.  If they're logged in and shouldn't be, log them out!

			$mmog_nologin = explode( ',', str_replace( ' ', '', $vbulletin->options['mmog_wiki_nologin_ug'] ) );

			foreach ( $mmog_nologin as $nologin ) {
			    if ( is_member_of( $vbulletin->userinfo, $nologin ) ) {
				$user->logout();
				break;
			}   }

                	return true;
        }	}

	if ( $vbulletin->userinfo['userid'] ) {
		// Check Nologin Groups - if they're a member then do not log them in.

		$mmog_nologin = explode( ',', str_replace( ' ', '', $vbulletin->options['mmog_wiki_nologin_ug'] ) );

		foreach ( $mmog_nologin as $nologin ) {
		    if ( is_member_of( $vbulletin->userinfo, $nologin ) )
			return true;
		}

		$username = $vbulletin->userinfo['username'];

		global $wgEnableParserCache, $wgParser, $wgOut, $wgCachePages;

		$wgEnableParserCache = false;
		if ($wgParser)
			$wgParser->disableCache();
		if ($wgOut)
			$wgOut->enableClientCache( false );
		$wgCachePages = false;

		if ( $username ) {
			$u = User::newFromName( $username );
			if ( $u->getID() == 0 ) {
				$u->addToDatabase();
				$u->setToken();
			} else {
			    $u->clearInstanceCache( 'id' );
			    $u->load();
			}

			$user = $u;
			$user->setOption( 'rememberpassword', 1 );
			$user->setupSession();
			$user->setCookies();
			$user->invalidateCache();
			$user->saveSettings();

			// Check Sysop Groups

			$mmog_sysopgroup = explode( ',', str_replace( ' ', '', $vbulletin->options['mmog_wiki_sysop_ug'] ) );

			foreach( $mmog_sysopgroup as $sysopgroup ) {
			    if ( is_member_of( $vbulletin->userinfo, $sysopgroup ) ) {
				$user->addGroup( 'sysop' );
				break;
			}   }

			$mmog_bureaugroup = explode( ',', str_replace( ' ', '', $vbulletin->options['mmog_wiki_bureaucrat_ug'] ) );

			foreach ( $mmog_bureaugroup as $bureaugroup ) {
			    if ( is_member_of( $vbulletin->userinfo, $bureaugroup ) ) {
				$user->addGroup( 'bureaucrat' );
				break;
			}   }

		}
	} else
		$user->logout();

	return true;
}

function vBLoginForm($template) {
    	    $template->data['link'] = 'Please use your forum information for login.  Login across the forum and wiki is the same.';

    	    // If there is an error message display it.
    	    if ($errorMessage)
    	    {
                $template->data['message'] = $errorMessage;
                $template->data['messagetype'] = 'error';
    	    }
    	    return true;
 }

function set_personal_urls( &$personal_urls, &$wgTitle ) {
	global $vbulletin;

	if ( $vbulletin->userinfo['userid'] !== 0 ) {

	$personal_urls['logout'] = array(
			'text' => wfMsg( 'userlogout' ),
			'href' => ( $vbulletin->options['bburl'] . '/login.php?do=logout&logouthash=' . $vbulletin->userinfo['logouthash'] ),
			'active' => false
			);

	}

	return TRUE;
}

// Shut down the vBulletin registry - we no longer need it.
// Do not save reporting for raw output files (CSS, images, etc)

if ( $_SERVER['REQUEST_METHOD'] === 'GET' ) {
    if ( strpos( $_SERVER['REQUEST_URI'], 'action=raw' ) === FALSE ) {
	exec_shut_down();
}   }

//$wgHooks['UserLogoutComplete'][] = 'QISSingleSignOnUserLogout';
if ( floatval( $wgVersion ) <= 1.12 )
	$wgHooks['AutoAuthenticate'][] = 'vBSSOAutoAuth';
else
	$wgHooks['UserLoadFromSession'][] = 'vBSSOAutoAuth';

$wgHooks['UserLoginForm'][] = 'vBLoginForm';
$wgHooks['PersonalUrls'][] = 'set_personal_urls';

$wgExtensionCredits['other'][] = array(
	'name' => 'MMOG Wiki',
	'version' => '2.1.1',
	'author' => 'Paul Stout',
	'url' => 'http://www.mymmogames.net/',
	'description' => 'Single Sign On from vBulletin'
);
