<?php

/**
 * @file bowob_functions.php
 *
 * Implements main functions and calls the BoWoB API.
 */

/**
 * Prints synchronization data to be readed by BoWoB server.
 * @return void.
 */
function bowob_server_sync()
{
  global $vbulletin;
  
  header('Cache-Control: no-cache, must-revalidate');
  header('Content-Type: text/plain; charset=utf-8');

  include_once(DIR . '/includes/bowob_api.php');

  echo bowob_api_get_sync_data(
    $vbulletin->input->clean_gpc('g', 'id', TYPE_INT),
    $vbulletin->input->clean_gpc('g', 'sync', TYPE_STR),
    $vbulletin->input->clean_gpc('g', 'name', TYPE_STR) == '1',
    $vbulletin->input->clean_gpc('g', 'avatar', TYPE_STR) == '1',
    $vbulletin->input->clean_gpc('g', 'friends', TYPE_STR) == '1'
  );
}

/**
 * Creates a synchronization record and prints record identifiers to be readed by BoWoB client.
 * @return void.
 */
function bowob_client_sync()
{
  global $vbulletin;
  
  header('Cache-Control: no-cache, must-revalidate');
  header('Content-Type: text/plain');

  if(!bowob_chat_allowed())
  {
    return;
  }

  include_once(DIR . '/includes/bowob_api.php');

  echo bowob_api_new_sync(
    $vbulletin->input->clean_gpc('g', 'nick', TYPE_STR),
    $vbulletin->input->clean_gpc('g', 'name', TYPE_STR) == '1',
    $vbulletin->input->clean_gpc('g', 'avatar', TYPE_STR) == '1'
  );
}

/**
 * Redirects to login page.
 * @return void.
 */
function bowob_redirect_login()
{
  global $vbulletin;
  
  exec_header_redirect('usercp.php' . $vbulletin->session->vars['sessionurl_q']);
}

/**
 * Redirects to user profile page.
 * @return void.
 */
function bowob_redirect_profile()
{
  global $vbulletin;
  
  $userid = $vbulletin->input->clean_gpc('g', 'id', TYPE_INT);
  
  exec_header_redirect('member.php?' . $vbulletin->session->vars['sessionurl'] . 'u=' . $userid);
}

/**
 * Gets BoWoB HTML code for show the chat.
 * @return string The HTML code.
 */
function bowob_code()
{
  global $vbulletin;
  
  if(!bowob_chat_allowed())
  {
    return '';
  }

  include_once(DIR . '/includes/bowob_api.php');

  return bowob_api_get_code(
    $vbulletin->options['bowob_app_id'],
    $vbulletin->options['bowob_server_address']
  );
}

/**
 * Checks if current user is logued.
 * @return boolean User is logued.
 */
function bowob_is_user_logued()
{
  global $vbulletin;

  if($vbulletin->userinfo['userid'] > 0)
  {
    return true;
  }
  else
  {
    return false;
  }
}

/**
 * Gets current user id.
 * @return int User id.
 */
function bowob_get_user_id()
{
  global $vbulletin;

  return $vbulletin->userinfo['userid'];
}

/**
 * Gets current user nick.
 * @return string User nick.
 */
function bowob_get_user_nick()
{
  global $vbulletin;

  if($vbulletin->userinfo['userid'] > 0)
  {
    return $vbulletin->userinfo['username'];
  }
  else
  {
    return '';
  }
}

/**
 * Gets current user name.
 * @return string User name.
 */
function bowob_get_user_name()
{
  return '';
}

/**
 * Gets current user avatar url.
 * @return string User avatar.
 */
function bowob_get_user_avatar()
{
  global $vbulletin;
  
  require_once(DIR . '/includes/functions_user.php');
  
  $url = fetch_avatar_url($vbulletin->userinfo['userid']);
  
  if($url)
  {
    return $vbulletin->options['bburl'] . '/' . $url[0];
  }
  else
  {
    return '';
  }
}

/**
 * Gets current user friends.
 * @param int $id User id.
 * @param string $separator Separator between nicks.
 * @return string User friends.
 */
function bowob_get_user_friends($id, $separator)
{
  global $vbulletin;
  
  if($id <= 0)
  {
    return '';
  }
  
  $output = '';
  
  $rs = $vbulletin->db->query_read("
    SELECT u.username
    FROM " . TABLE_PREFIX . "user u, " . TABLE_PREFIX . "userlist l
    WHERE u.userid = l.relationid AND l.userid = " . intval($id) . " AND l.friend = 'yes'
  ");
  if(!$rs)
  {
    return '';
  }
  
  while($rw = $vbulletin->db->fetch_row($rs))
  {
    $output .= $rw[0] . $separator;
  }

  return $output;
}

/**
 * Stores a synchronization record in database.
 * @param string $auth Record auth.
 * @param int $creation Record creation time.
 * @param int $user_id Record user id.
 * @param string $user_nick Record user nick.
 * @param string $user_name Record user name.
 * @param string $user_avatar Record user avatar.
 * @param int $user_type Record user type.
 * @return int Record id.
 */
function bowob_store_sync($auth, $creation, $user_id, $user_nick, $user_name, $user_avatar, $user_type)
{
  global $vbulletin;
  
  $vbulletin->db->query_write("
    INSERT INTO " . TABLE_PREFIX . "bowob
    (
      auth,
      creation,
      user_id,
      user_nick,
      user_name,
      user_avatar,
      user_type
    )
    VALUES
    (
      '" . $vbulletin->db->escape_string($auth) . "',
      " . intval($creation) . ",
      " . intval($user_id) . ",
      '" . $vbulletin->db->escape_string($user_nick) . "',
      '" . $vbulletin->db->escape_string($user_name) . "',
      '" . $vbulletin->db->escape_string($user_avatar) . "',
      " . intval($user_type) . "
    )
  ");
  
  return $vbulletin->db->insert_id();
}

/**
 * Extracts a synchronization record from database.
 * @param int $id Record id.
 * @param string $auth Record auth.
 * @param int $expiration Record expiration time.
 * @return array Record values.
 */
function bowob_extract_sync($id, $auth, $expiration)
{
  global $vbulletin;
  
  $vbulletin->db->query_write("
    DELETE FROM " . TABLE_PREFIX . "bowob
    WHERE creation < " . intval($expiration) . "
  ");
  
  $rs = $vbulletin->db->query_read("
    SELECT auth, user_id, user_nick, user_name, user_avatar, user_type
    FROM " . TABLE_PREFIX . "bowob
    WHERE id = " . intval($id) . "
  ");
  if(!$rs)
  {
    return array();
  }
  
  $rw = $vbulletin->db->fetch_array($rs);
  if(!$rw OR $rw['auth'] != $auth)
  {
    return array();
  }
  else
  {
    $vbulletin->db->query_write("
      DELETE FROM " . TABLE_PREFIX . "bowob
      WHERE id = " . intval($id) . "
    ");

    return array(
      'user_id' => $rw['user_id'],
      'user_nick' => $rw['user_nick'],
      'user_name' => $rw['user_name'],
      'user_avatar' => $rw['user_avatar'],
      'user_type' => $rw['user_type'],
    );
  }
}

/**
 * Checks if current user / page is allowed to chat.
 * @return boolean chat is allowed.
 */
function bowob_chat_allowed()
{
  global $vbulletin;
  
  $pages_restricted = $vbulletin->options['bowob_pages'];
  $groups_restricted = $vbulletin->options['bowob_usergroups'];
  
  $pages_allowed = true;
  $groups_allowed = true;
  
  if(!empty($pages_restricted))
  {
    $cp = explode('?', basename($_SERVER['REQUEST_URI']));
    $pages_allowed = !in_array($cp[0], explode(',', $pages_restricted));
  }
  
  if(!empty($groups_restricted))
  {
    foreach(explode(',', $groups_restricted) AS $ug)
    {
      if(is_member_of($vbulletin->userinfo, $ug))
      {
        $groups_allowed = false;
        break;
      }
    }
  }
  
  return ($pages_allowed AND $groups_allowed);
}

?>
