<?php

#error_reporting(E_ALL);

/**
	@author Andreas "Radon" Rudolph
	@author Disasterpiece (vbulletin.org)
	@url http://purgatory-labs.de
	@url http://vbulletin.org
	@date last-modified: 17/05/2011

	Distribution notes

	This class may not be redistributed without written permission to do so
	and *must be left unmodified*, including this information block!
	Written as part as a vbulletin modification


**/
class pl_image_manipulation {

	var $registry = null;
	var $options = null;

	var $imageinfo = array();
	var $imagedata = array();

	var $supported_extensions = array("jpg", "jpeg", "gif", "png");

	/**
		Constructor, initializes registry
		@param $registry: the $vbulletin var
	**/
	function pl_image_manipulation(&$registry, &$options=null) {
		$this->registry = $registry;
		$this->options = isset($options) ? $options : $registry->options;
	}


	/**
		Adds a raw imageresource to the layerset
		@param $imageresource: the resource created with imagecreate*
		@param $extension: data extension of type (jpg, jpeg, gif, png)
		@return int: new index of added layer
	**/
	function addRawImage($imageresource, $extension) {
		$iw = imagesx($imageresource);
		$ih = imagesy($imageresource);
		
		if (!in_array($extension, $this->supported_extensions))
			throw new Exception("TODO");
		
		$newimageinfo = array("extension" => $extension, "height" => $ih, "width" => $iw);
		

		$newimagedata = $imageresource;
		
		if (empty($newimageinfo) || empty($newimagedata))
			throw new Exception("pl_image_manipulation: Image data or image info corrupt!");
			
		if ($newimageinfo['extension'] == 'png'
			|| $newimageinfo['extension'] == 'gif') {
			
			// Preserve alpha channel if picture supports transparency
			imagealphablending($newimagedata, true);
			imagesavealpha($newimagedata, true);
		}

		$this->imageinfo[] = $newimageinfo;
		$this->imagedata[] = $newimagedata;
		
		return count($this->imageinfo)-1;
	}


	/**
		Adds an imagefile to the list of manipulated images for later combination
		@param $path: absolute path to the image file
		@return int: new index of added layer
	**/
	function addFileImage($path) {
		#echo "addImage (".print_r($path,1).")\n";

		$tmp = explode(".",$path);
		$extension = end($tmp);
		unset($tmp);
		
		list($iw, $ih, $type, $attr) = getimagesize($path);
		$newimageinfo = array("extension" => $extension, "height" => $ih, "width" => $iw);
		
		$s = file_get_contents($path);
		if (!$s) return false;
		$newimagedata = imagecreatefromstring($s);
		
		if (empty($newimageinfo) || empty($newimagedata))
			throw new Exception("pl_image_manipulation: Image data or image info corrupt!");
			
		if ($newimageinfo['extension'] == 'png'
			|| $newimageinfo['extension'] == 'gif') {
			
			// Preserve alpha channel if picture supports transparency
			imagealphablending($newimagedata, true);
			imagesavealpha($newimagedata, true);
		}

		$this->imageinfo[] = $newimageinfo;
		$this->imagedata[] = $newimagedata;
		
		return count($this->imageinfo)-1;
	}


	/**
		Adds an image to the list of manipulated images for later combination, resizing, etc.
		@param $attachmentinfo: array in the form of standard attachmentinfo.
		@return int: new index of added layer
		@see attachment.php for format requirements
	**/
	function addAttachmentImage($attachmentinfo) {

   		if (!isset($attachmentinfo['attachmentid']))
			return false;
		#var_dump($attachmentinfo);

		$vbulletin = &$this->registry;
		$db = &$vbulletin->db;

		// Retrieve additional info if we just provided the id
		if (!isset($attachmentinfo['filedataid']) || !isset($attachmentinfo['uploader'])) {

			$this->loadFileInfo($attachmentinfo);

		}
		#var_dump($attachmentinfo);

		$newimageinfo = &$attachmentinfo;

		// Save method: Filesystem
		if ($this->options['attachfile']) {

			require_once(DIR . '/includes/functions_file.php');
			$attachpath = fetch_attachment_path($attachmentinfo['uploader'], $attachmentinfo['filedataid']);

			$newimagedata = imagecreatefromstring(file_get_contents($attachpath));
			$newimageinfo['width'] = imagesx($newimagedata);
			$newimageinfo['height'] = imagesy($newimagedata);

		}
		// Save method: Database
		else {

        	if (!isset($attachmentinfo['filedata']))
				$this->loadFileData($attachmentinfo);
			$newimageinfo = $attachmentinfo;

			$newimagedata = imagecreatefromstring($attachmentinfo['filedata']);
			// Altough database says otherwise, the actual image size may be changed,
			// because the attachment display code seems to resize the picture before
			// displaying it. So we can't trust the database value.
			$newimageinfo['width'] = imagesx($newimagedata);
			$newimageinfo['height'] = imagesy($newimagedata);

		}

		if (empty($newimageinfo) || empty($newimagedata))
			throw new Exception("pl_image_manipulation: Image data or image info corrupt!");

		if ($newimageinfo['extension'] == 'png'
			|| $newimageinfo['extension'] == 'gif') {
			
			// Preserve alpha channel if picture supports transparency
			imagealphablending($newimagedata, true);
			imagesavealpha($newimagedata, true);
		}

		$this->imageinfo[] = $newimageinfo;
		$this->imagedata[] = $newimagedata;
		
		return count($this->imageinfo)-1;
	}
	
	/**
		Combines all image layers into 1 final image. Base size is image @ index 0
		@return boolean: true/false weather all layers have been applied succesfully
	**/
	function flattenLayers() {
		#echo "flattenLayers ()\n";

		$n = count($this->imageinfo);
		if ($n < 2) return; // Already consists of 1 image layer

		$res = true;
		for ($i=1 ; $i<$n ; $i++) {

			$this->alpha_blending(
				$this->imagedata[0],
				$this->imagedata[$i],
				isset($this->imageinfo[$i]['positioninfo'][0]) ? $this->imageinfo[$i]['positioninfo'][0] : ($this->imageinfo[0]['width'] - $this->imageinfo[$i]['width']),
				isset($this->imageinfo[$i]['positioninfo'][1]) ? $this->imageinfo[$i]['positioninfo'][1] : ($this->imageinfo[0]['height'] - $this->imageinfo[$i]['height']),
				$this->imageinfo[$i]['width'],
				$this->imageinfo[$i]['height']
			);

		} # end for

		// Rebuild layer arrays
		$shift_info = $this->imageinfo[0];
		$shift_data = $this->imagedata[0];
		$this->imageinfo = array($shift_info);
		$this->imagedata = array($shift_data);

		return $res;
	}
	

	/**
		Adds a transparent image to a base layer, preserving its transparency
		@param $dest: the base layer image handler
		@param $source: the transparent image to put on the base layer
		@param $offset_x: gap left border <-> image
		@param $offset_y: gap top border <-> image
		@param $width: width of source picture
		@param $height: height of source picture
		@see http://de3.php.net/manual/en/function.imagealphablending.php#71314
	**/
	function alpha_blending ($dest, $source, $offset_x, $offset_y, $width, $height) {

		for ($x=0 ; $x<$width ; $x++) {
			for ($y=0 ; $y<$height ; $y++) {

                $src_col = imagecolorat($source, $x ,$y);
                $dst_col = imagecolorat($dest, $x+$offset_x, $y+$offset_y);

                $a_s = ($src_col >> 24) << 1;
                $r_s = $src_col >> 16 & 0xFF;
                $g_s = $src_col >> 8 & 0xFF;
                $b_s = $src_col & 0xFF;
                $r_d = $dst_col >> 16 & 0xFF;
                $g_d = $dst_col >> 8 & 0xFF;
                $b_d = $dst_col & 0xFF;

                if ($a_s == 0) {
                    $r_d = $r_s;
					$g_d = $g_s;
					$b_d = $b_s;
                }
                else if ($a_s > 253) {
                } else {
                    $r_d = (($r_s * (0xFF-$a_s)) >> 8) + (($r_d * $a_s) >> 8);
                    $g_d = (($g_s * (0xFF-$a_s)) >> 8) + (($g_d * $a_s) >> 8);
                    $b_d = (($b_s * (0xFF-$a_s)) >> 8) + (($b_d * $a_s) >> 8);
                }
                $rgb_d = imagecolorallocatealpha($dest, $r_d, $g_d, $b_d, 0);
                imagesetpixel($dest, $x+$offset_x, $y+$offset_y, $rgb_d);
            }
        }
    }

	
	/**
		Adds position info to the index layer like position, size, offset
		@param $index: index of the image to apply positioninfo to
		@param $positioninfo: The position offset array( int $dst_x , int $dst_y )
	**/
	function setPositionInfo($index, $positioninfo) {

		if (!@is_array($positioninfo) || count($positioninfo) < 2)
			throw new Exception("pl_image_manipulation: Malformed positionInfo, cannot set these parameters!");

		return ($this->imageinfo[$index]['positioninfo'] = $positioninfo);

	}


	/**
		Clears all layers
	**/
	function clearLayers() {
		foreach ($this->imagedata as $id) imagedestroy($id);
		$this->imageinfo = array();
		$this->imagedata = array();
	}


	/**
		Ensures that this image gets @ position 0 as base layer
		@param $index: the index which will switch place with index 0
	**/
	function forceBaseLayerIndex($index) {
		$this->imageinfo[0] ^= $this->imageinfo[$index];
		$this->imageinfo[$index] ^= $this->imageinfo[0];
		$this->imageinfo[0] ^= $this->imageinfo[$index];
	}
	
	
	/**
		Ensures that this image gets @ position 0 as base layer
		@param $attachmentinfo: the image which will be the new base layer
		@return bool: true/false weather the addImage function was successful
			(may be false if attachmentinfo was malformed)
	**/
	function forceBaseLayerAttachment($attachmentinfo) {
		$new_index = count($this->imageinfo);
		if (!$this->addAttachmentImage($attachmentinfo)) return false;
		$this->forceBaseLayerIndex($new_index);
		return true;
	}


	/**
		Ensures that this image gets @ position 0 as base layer
		@param $path: the absolute path of the file to add to position 0
		@return bool: true/false weather the addImage function was successful
			(may be false if file wasn't found or inaccessible)
	**/
	function forceBaseLayerFile($path) {
		$new_index = count($this->imageinfo);
		if (!$this->addImage($path)) return false;
		$this->forceBaseLayerIndex($new_index);
		return true;
	}


	/**
		Returns image content as string
		@param $print: switch for printing image to screen
		@param $headers: switch if headers should be passed too (in case they don't have been generated before)
		@return string/bool: image content or true, if print is enabled
	**/
	function getImageString($print=false, $headers=false) {
		$vbulletin = &$this->registry;

		ob_end_flush(); # Destroy previous ob if there is any
		if (!$print) {
	        ob_start();
		}
        switch ($this->imageinfo[0]['extension']) {
			case 'jpg':
			case 'jpeg':
				if ($headers) header('Content-Type: image/jpeg');
			    imagejpeg($this->imagedata[0], null, (int)$this->options['plwm_watermark_quality']);
			    break;

			case 'png':
				if ($headers) header('Content-Type: image/png');
				imagepng($this->imagedata[0], null, intval(($this->options['plwm_watermark_quality'] - 100)) / 11.111111, PNG_NO_FILTER);
				break;

			case 'gif':
				if ($headers) header('Content-Type: image/gif');
				imagegif($this->imagedata[0]);
				break;

			default:
				ob_end_clean();
				return "";
		}
	    return ob_get_clean();
	}


	/**
		Saves image content to file
		@param $newpath: new (absolute) path to save the image to
		@return true/false if save was successful
	**/
	function saveAsFile($newpath) {
		$vbulletin = &$this->registry;

        switch ($this->imageinfo[0]['extension']) {
			case 'jpg':
			case 'jpeg':
			    return imagejpeg($this->imagedata[0], $newpath, (int)$this->options['plwm_watermark_quality']);

			case 'png':
				return imagepng($this->imagedata[0], $newpath, intval(($this->options['plwm_watermark_quality'] - 100) / 11.111111), PNG_NO_FILTER);

			case 'gif':
				return imagegif($this->imagedata[0], $newpath);

			default:
				return false;
		}
	}


	/**
	    Adds a text string with the specified parameters to the specified layer
		@param $string: the string
		@param $font: which font the string should use
		@param $alpha: alpha value
		@param $color: hex-representation of colorcode in css-like string-form ("#FF0000" eq. red)
		@param $shadecolor: color of the text-shade. "transparent" or "none" if no shade should be painted.
		@param $index: Index of the layer to paint the string on
		@return bool: true/false - the result of the imagestring function
	**/
	function stringToLayer($string, $font, $color, $shadecolor, $alpha, $pos, $index=0) {

		// css-style hex value to real hex conversion #123456 -> 0x12, 0x34, 0x56
		$z = ($color{0}=='#') ? 1 : 0;
		$short = (strlen($color) <= 4);
		$c_r = hexdec( $short ? $color{$z} : $color{$z}.$color{$z+1} );
		$c_g = hexdec( $short ? $color{$z+1} : $color{$z+2}.$color{$z+3} );
		$c_b = hexdec( $short ? $color{$z+2} : $color{$z+4}.$color{$z+5} );
		// shadecolor
		$z = ($shadecolor{0}=='#') ? 1 : 0;
		$short = (strlen($shadecolor) <= 4);
		$s_r = hexdec( $short ? $shadecolor{$z} : $shadecolor{$z}.$shadecolor{$z+1} );
		$s_g = hexdec( $short ? $shadecolor{$z+1} : $shadecolor{$z+2}.$shadecolor{$z+3} );
		$s_b = hexdec( $short ? $shadecolor{$z+2} : $shadecolor{$z+4}.$shadecolor{$z+5} );

		list($iw, $ih) = $this->getLayerSize(0);
		$ww = imagefontwidth($font) * strlen($string);
		$wh = imagefontheight($font);
		$pos = explode(',',$pos);
		if (!isset($pos[1])) $pos[1] = 0;
		if (!isset($pos[2])) $pos[2] = 0;

		// position translation
		switch ($pos[0]) {

			case 0: // Upper left corner
				$o_x = $pos[1];
				$o_y = $pos[2];
				break;

			case 1: // Upper right corner
				$o_x = $iw - $ww - $pos[1];
				$o_y = $pos[2];
				break;

			case 2: // Bottom right corner
				$o_x = $iw - $ww - $pos[1];
				$o_y = $ih - $wh - $pos[2];
	            break;

			case 3: // Bottom left corner
				$o_x = $pos[1];
				$o_y = $ih - $wh - $pos[2];
				break;

			case 4: // Center
			default:
				$o_x = intval(($iw/2) - ($ww/2) - $pos[1]);
				$o_y = intval(($ih/2) - ($wh/2) - $pos[2]);

		}

		// Calculate alpha value from percent
		$alpha = (100 - min(100,abs($alpha))) * 1.27;
		// Shade font
		$shadecol = imagecolorallocatealpha($this->imagedata[$index], $s_r, $s_g, $s_b, min(127,($alpha*1.2)));
		$res = (is_int(imagestring($this->imagedata[$index], $font, $o_x+2, $o_y+2, $string, $shadecol)));
		// String
		$strcol = imagecolorallocatealpha($this->imagedata[$index], $c_r, $c_g, $c_b, $alpha);
		$res &= (is_int(imagestring($this->imagedata[$index], $font, $o_x, $o_y, $string, $strcol)));
		return $res;
	}
	
	/**
	    Adds a text string with the specified parameters to the specified layer
		@param $string: the string
		@param $ttffile: font file
		@param $size: font-size of the string
		@param $color: hex-representation of colorcode in css-like string-form ("#FF0000" eq. red)
		@param $shadecolor: color of the text-shade. "transparent" or "none" if no shade should be painted.
		@param $alpha: alpha value
		@param $pos: position info
		@param $index: Index of the layer to paint the string on
		@return bool: true/false - the result of the imagestring function
	**/
	function stringTTFToLayer($string, $ttffile, $size, $color, $shadecolor, $alpha, $pos, $index=0) {
        // css-style hex value to real hex conversion #123456 -> 0x12, 0x34, 0x56
		$z = ($color{0}=='#') ? 1 : 0;
		$short = (strlen($color) <= 4);
		$c_r = hexdec( $short ? $color{$z}.$color{$z} : $color{$z}.$color{$z+1} );
		$c_g = hexdec( $short ? $color{$z+1}.$color{$z+1} : $color{$z+2}.$color{$z+3} );
		$c_b = hexdec( $short ? $color{$z+2}.$color{$z+2} : $color{$z+4}.$color{$z+5} );
		// Shade color
		$z = ($shadecolor{0}=='#') ? 1 : 0;
		$short = (strlen($shadecolor) <= 4);
		$s_r = hexdec( $short ? $shadecolor{$z}.$shadecolor{$z} : $shadecolor{$z}.$shadecolor{$z+1} );
		$s_g = hexdec( $short ? $shadecolor{$z+1}.$shadecolor{$z+1} : $shadecolor{$z+2}.$shadecolor{$z+3} );
		$s_b = hexdec( $short ? $shadecolor{$z+2}.$shadecolor{$z+2} : $shadecolor{$z+4}.$shadecolor{$z+5} );

		list($iw, $ih) = $this->getLayerSize(0);
		$ittfbb = imagettfbbox($size, 0, $ttffile, $string);
		$ww = $ittfbb[4] - $ittfbb[6];
		$wh = abs($ittfbb[7] - $ittfbb[1]);
		#var_dump($ittfbb);
		$pos = explode(',',$pos);
		if (!isset($pos[1])) $pos[1] = 0;
		if (!isset($pos[2])) $pos[2] = 0;

		// position translation
		switch ($pos[0]) {

			case 0: // Upper left corner
				$o_x = $pos[1];
				$o_y = $pos[2] + $wh;
				break;

			case 1: // Upper right corner
				$o_x = $iw - $ww - $pos[1];
				$o_y = $pos[2] + $wh;
				break;

			case 2: // Bottom right corner
				$o_x = $iw - $ww - $pos[1];
				$o_y = $ih - $pos[2];
	            break;

			case 3: // Bottom left corner
				$o_x = $pos[1];
				$o_y = $ih - $pos[2];
				break;

			case 4: // Center
			default:
				$o_x = intval(($iw/2) - ($ww/2) - $pos[1]);
				$o_y = intval(($ih/2) - ($wh/2) - $pos[2]);

		}

		// Calculate alpha value from percent
		$alpha = (100 - min(100,abs($alpha))) * 1.27;
		// Shade font
		$shadecol = imagecolorallocatealpha($this->imagedata[$index], $s_r, $s_g, $s_b, min(127,($alpha*1.2)));
		$res = (is_array(imagettftext($this->imagedata[$index], $size, 0, $o_x+2, $o_y+2, $shadecol, $ttffile, $string)));
		// String
		$strcol = imagecolorallocatealpha($this->imagedata[$index], $c_r, $c_g, $c_b, $alpha);
		$res &= (is_array(imagettftext($this->imagedata[$index], $size, 0, $o_x, $o_y, $strcol, $ttffile, $string)));
		return $res;
	}


	/**
		Returns width, height of the specified layer
		@return array(int, int): width, height of the layer
	**/
	function getLayerSize($i) {
		return array($this->imageinfo[$i]['width'], $this->imageinfo[$i]['height']);
	}

	
	/**
		Loads file data from vb filedata table and adds to attachmentinfo
		@param $attachmentinfo
	**/
	function loadFileData(&$attachmentinfo, $loadFiledata=true) {
		$db = &$this->registry->db;
		$query = "
				SELECT filedataid, ".($loadFiledata ? "filedata, " : "")."filesize, extension, height, width
				FROM " . TABLE_PREFIX . "filedata
				WHERE filedataid = $attachmentinfo[filedataid]
			";

		$res = $db->query_first_slave($query);
		foreach ($res as $key => $val) {
			$attachmentinfo[$key] = $val;
		}
	}

	/**
		Loads file info from vb attachment table and adds to attachmentinfo
		@param $attachmentinfo
	**/
	function loadFileInfo(&$attachmentinfo) {
		$db = &$this->registry->db;
		$query = "
				SELECT contenttypeid, contentid, userid, filedataid
				FROM " . TABLE_PREFIX . "attachment
				WHERE attachmentid = $attachmentinfo[attachmentid]
			";

		$res = $db->query_first_slave($query);
		foreach ($res as $key => $val) {
			$attachmentinfo[$key] = $val;
		}
	}

}