<?php
/**
 *	     SteamProfile.
 *
 *	 : Nico Bergemann <barracuda415@yahoo.de>
 *	Copyright 2009 Nico Bergemann
 *
 *         : Gl0BuZ
 *      <globuz@no-steam.ru>
 *      Copyright 2009 Gl0BuZ
 *
 *       : Alex 'Gl0BuZ' Shchepilov
 *      <globuz@no-steam.ru>
 *      Copyright 2010 Gl0BuZ
 *
 */

class HTTPHeaders {
	private static $aRequestHeaders = array();
	private static $aResponseHeaders = array();
	private static $aResponseCodes = array(
		100 => 'Continue',
		101 => 'Switching Protocols',
		200 => 'OK',
		201 => 'Created',
		202 => 'Accepted',
		203 => 'Non-Authoritative Information',
		204 => 'No Content',
		205 => 'Reset Content',
		206 => 'Partial Content',
		300 => 'Multiple Choices',
		301 => 'Moved Permanently',
		302 => 'Found',
		303 => 'See Other',
		304 => 'Not Modified',
		305 => 'Use Proxy',
		307 => 'Temporary Redirect',
		400 => 'Bad Request',
		401 => 'Unauthorized',
		402 => 'Payment Required',
		403 => 'Forbidden',
		404 => 'Not Found',
		405 => 'Method Not Allowed',
		406 => 'Not Acceptable',
		407 => 'Proxy Authentication Required',
		408 => 'Request Timeout',
		409 => 'Conflict',
		410 => 'Gone',
		411 => 'Length Required',
		412 => 'Precondition Failed',
		413 => 'Request Entity Too Large',
		414 => 'Request-URI Too Long',
		415 => 'Unsupported Media Type',
		416 => 'Requested Range Not Satisfiable',
		417 => 'Expectation Failed',
		500 => 'Internal Server Error',
		501 => 'Not Implemented',
		502 => 'Bad Gateway',
		503 => 'Service Unavailable',
		504 => 'Gateway Timeout',
		505 => 'HTTP Version Not Supported'
	);
	private static $aSpecialCase = array(
		'te' 				=> 'TE',
		'content-md5' 		=> 'Content-MD5',
		'etag' 				=> 'ETag',
		'www-authenticate'	=> 'WWW-Authenticate'
	);
	
	public function __construct() {
		if(count(self::$aRequestHeaders) == 0) {
			$this->refreshRequestHeaders();
		}
		
		if(count(self::$aResponseHeaders) == 0) {
			$this->refreshResponseHeaders();
		}
	}
	
	public function formatKey($sKey, $sSeparator) {
		$sKey = trim($sKey);
	

		$sKeyLower = strtolower($sKey);
		if(isset(self::$aSpecialCase[$sKeyLower])) {
			return self::$aSpecialCase[$sKeyLower];
		}
	

		$aWords = explode($sSeparator, $sKey);
		$iWords = count($aWords);
		

		for($i = 0; $i < $iWords; $i++) {
			$aWords[$i] = $this->fixKeyCase($aWords[$i]);
		}
		

		return implode('-', $aWords);
	}
	
	public function fixKeyCase($sKey) {
		return ucfirst(strtolower($sKey));
	}
	
	public function refreshRequestHeaders() {

		self::$aRequestHeaders = array();
	

		if(function_exists('apache_request_headers')) {
			$aRequestHeaders = apache_request_headers();
			

			foreach($aRequestHeaders as $sKey => $sVal) {
				$sKey = $this->formatKey($sKey, '-');
				self::$aRequestHeaders[$sKey] = $sVal;
			}
		} else {
			foreach($_SERVER as $sKey => $sVal) {

				if(substr($sKey, 0, 5) != 'HTTP_') {
					continue;
				}

				$sKey = $this->formatKey(substr($sKey, 5), '_');
				self::$aRequestHeaders[$sKey] = $sVal;
			}
		}
	}
	
	public function refreshResponseHeaders() {

		self::$aResponseHeaders = array();
	

		if(function_exists('apache_response_headers')) {
			self::$aResponseHeaders = apache_response_headers();
		} else {
			$aHeaderList = headers_list();
			foreach($aHeaderList as $sHeader) {
				$aHeader = explode(':', $sHeader);
				self::$aResponseHeaders[$aHeader[0]] = trim($aHeader[1]);
			}
		}
	}
	
	public function setResponseCode($iCode) {
		if(isset(self::$aResponseCodes[$iCode])) {
			return $this->setResponse('HTTP/1.1 '.$iCode.' '.self::$aResponseCodes[$iCode]);
		} else {
			return false;
		}
	}

	public function setResponse($sName, $sValue = null, $bReplace = true) {
		if(headers_sent()) {
			return false;
		} else {
			$sName = $this->formatKey($sName, '-');
			
			if($sValue == null) {
				header($sName, $bReplace);
			} else {
				header("$sName: $sValue", $bReplace);
				self::$aResponseHeaders[$sName] = $sValue;
			}
			
			return true;
		}
	}
	
	public function getResponse($sName) {
		return isset(self::$aResponseHeaders[$sName])? self::$aResponseHeaders[$sName] : null;
	}
	
	public function getResponseAll() {
		return self::$aResponseHeaders;
	}
	
	public function setRedirect($sTarget, $bRelative = true) {
		$sHost = $_SERVER['HTTP_HOST'];
		$sUri = $bRelative? dirname($_SERVER['PHP_SELF']) : '';
		

		$this->setResponseCode(303);
		$this->setResponse('Location', "http://$sHost$sUri/$sTarget");
	}
	
	public function getRequest($sName) {
		return isset(self::$aRequestHeaders[$sName])? self::$aRequestHeaders[$sName] : null;
	}
	
	public function getRequestAll() {
		return self::$aRequestHeaders;
	}
	
	public function isModifiedSince($iTime) {
		$sModifiedSet = $this->getRequest('If-Modified-Since');
		$sModifiedActual = gmdate('D, d M Y H:i:s \G\M\T', $iTime);
		
		if($sModifiedSet == null) {
			$this->setResponse('Last-Modified', $sModifiedActual);
			return true;
		}
		
		if($sModifiedSet === $sModifiedActual) {
			return false;
		} else {
			$this->setResponse('Last-Modified', $sModifiedActual);
			return true;
		}
	}
	
	public function hasEntityTag($sETag) {
		$sCurrentETag = $this->getRequest('If-None-Match');
		
		if($sCurrentETag == null || $sCurrentETag !== $sETag) {
			$this->setResponse('ETag', $sETag);
			return false;
		} else {
			return true;
		}
	}
}
?>
