<?php

/**
 * Permission model for instances.
 *
 * @package DBTech_xFShout
 */
class DBTech_xFShout_Model_InstancePermissions extends XenForo_Model
{
	/**
	 * Gets the phrase name for a permission.
	 *
	 * @param string $permissionId
	 *
	 * @return string
	 */
	public function getPermissionTitlePhraseName($permissionId)
	{
		return 'dbtech_xfshout_permission_' . $permissionId;
	}
	
	/**
	 * Prepares a permission for display.
	 *
	 * @param array $permission
	 *
	 * @return array
	 */
	public function preparePermission(array $permission)
	{
		$permission['title'] = new XenForo_Phrase($this->getPermissionTitlePhraseName(
			$permission['permission_id']
		));

		return $permission;
	}
	
	/**
	 * Prepares a list of permissions for display.
	 *
	 * @param array $permissions Format: [] => permission info
	 *
	 * @return array
	 */
	public function preparePermissions(array $permissions)
	{
		$retval = array();
		foreach ($permissions AS &$permission)
		{
			$permission = $this->preparePermission($permission);
			$retval[$permission['permission_id']] = $permission;
		}

		return $retval;
	}	
	/**
	 * Prepares a list of user permissions for display.
	 *
	 * @param array $permissions Format: [user_group_id][permission_id] => permission value
	 *
	 * @return array
	 */
	public function prepareUserPermissions(array $permissions)
	{
		$retval = array();
		foreach ($permissions AS $permission)
		{
			$retval[$permission['permission_id']] = $permission['value'];
		}

		return $retval;
	}
	
	/**
	 * Sets all permissions.
	 * 
	 * @param integer Usergroup ID
	 * @param string Permission ID
	 * @param integer Instance ID
	 * @param integer 1 = Yes / 0 = No
	 *
	 * @return boolean True
	 */
	public function setPermission($userGroupId, $permission_id, $instance_id, $yesno)
	{	
		// Not MVC enough for you? I don't care.
		$this->_getDb()->query("
			REPLACE INTO xf_dbtech_xfshout_permission_entry
				(user_group_id, permission_id, instance_id, permission_value)
			VALUES (
				'" . intval($userGroupId) . "',
				" . $this->_getDb()->quote($permission_id) . ",
				'" . intval($instance_id) . "',
				'" . intval($yesno) . "'
			)
		");
		
		return true;
	}
	
	/**
	 * Sets all permissions.
	 * 
	 * @param integer Usergroup ID
	 * @param integer Instance ID
	 * @param array Array of permissions for this usergroup
	 *
	 * @return boolean True
	 */
	public function setAllPermissions($userGroupId, $instanceId, $permissions)
	{
		foreach ($permissions as $permission_id => $yesno)
		{
			// Set the permission
			$this->setPermission($userGroupId, $permission_id, $instanceId, $yesno);
		}		
		
		return true;
	}	
	
	/**
	 * Get all permissions, in their relative display order.
	 *
	 * @return array Format: [] => permission info
	 */
	public function getAllPermissions()
	{
		return $this->_getDb()->fetchAll('
			SELECT *,
				default_value AS value
			FROM xf_dbtech_xfshout_permission
			ORDER BY display_order
		');
	}
	
	/**
	 * Gets all permissions in their relative display order, with the correct/effective
	 * value for the specified user group or user.
	 *
	 * @param integer $userGroupId
	 *
	 * @return array Format: [] => permission id / value
	 */
	public function getAllPermissionsWithValues($instanceId = 0, $userGroupId = 0)
	{
		$userGroupId = intval($userGroupId);
		$instanceId = intval($instanceId);
		
		return $this->_getDb()->fetchAll('
			SELECT 
				permission.permission_id,
				? AS user_group_id,
				COALESCE(entry.permission_value, permission.default_value) AS value
			FROM xf_dbtech_xfshout_permission AS permission
			LEFT JOIN xf_dbtech_xfshout_permission_entry AS entry ON(
				entry.permission_id = permission.permission_id
					AND entry.user_group_id = ?
					AND entry.instance_id = ?
			)
			ORDER BY permission.display_order
		', array($userGroupId, $userGroupId, $instanceId));
	}
	
	/**
	 * Gets all permissions, with values
	 * for the permissions specified for a particular group, or all groups.
	 *
	 * @param integer $userGroupId
	 *
	 * @return array Format: [user_group_id] => permission info with effective value
	 */
	public function getUserCollectionPermissionsForInterface($instanceId = 0, $userGroupId = 0)
	{
		$permissions = array();
		if ($userGroupId)
		{
			// Just one user group id
			$permissions[$userGroupId] = $this->prepareUserPermissions($this->getAllPermissionsWithValues($instanceId, $userGroupId));
		}
		else
		{
			foreach ($this->getModelFromCache('XenForo_Model_UserGroup')->getAllUserGroups() as $userGroup)
			{
				// All user groups
				$permissions[$userGroup['user_group_id']] = array(
					'title' => $userGroup['title'],
					'permissions' => $this->prepareUserPermissions($this->getAllPermissionsWithValues($instanceId, $userGroup['user_group_id']))
				);
			}
		}
		
		return $permissions;
	}
	
	/**
	 * Gets all permissions
	 *
	 * @return array Format: [permission_id] => permission info with effective value
	 */
	public function getCollectionPermissionsForInterface()
	{
		return $this->preparePermissions($this->getAllPermissions());
	}
	

	/**
	 * Rebuilds the instance permissions cache.
	 *
	 * @return array
	 */
	public function rebuildInstancePermissionsCache()
	{
		$cache = $this->_getDb()->fetchAll('
			SELECT 
				permission.permission_id,
				entry.*,
				COALESCE(entry.permission_value, permission.default_value) AS value
			FROM xf_dbtech_xfshout_permission AS permission
			LEFT JOIN xf_dbtech_xfshout_permission_entry AS entry ON(
				entry.permission_id = permission.permission_id
			)
			ORDER BY permission.display_order
		');
		
	    $arr = array();
    	foreach ($cache as $permission)
    	{
    		$arr[$permission['instance_id']][$permission['user_group_id']][$permission['permission_id']] = $permission['value'];
    	}		

		$this->_getDataRegistryModel()->set('dbtech_xfshout_instperms', $arr);
		return $cache;
	}	
}