<?php
/* -*- mode: php; tab-width: 4; indent-tabs-mode: t; coding: utf-8 -*- */

/**
 * vbulletin add-on fussball-tippspiele
 *
 * php version 5
 * vbulletin version >= 4.0.2
 *         _
 *  __   _| |__  ___  ___   ___ ___ ___ _ __
 *  \ \ / / '_ \/ __|/ _ \ / __/ __/ _ \ '__|
 *   \ V /| |_) \__ \ (_) | (_| (_|  __/ |
 *    \_/ |_.__/|___/\___/ \___\___\___|_|
 *
 * @package vbsoccer
 * @author  aa    <aa@geb-team.de>
 * @version SVN: $Id: vbsoccer_services.php 1352 2010-05-09 18:06:45Z aa $
 */

/** vBulletin XML objects */
require_once DIR . '/includes/class_xml.php';

/** vBulletin remote url */
require_once DIR . '/includes/class_vurl.php';

/**
* vbsoccer xml-rpc client
*
* @package vbsoccer
*/
class vBSoccer_XMLRPC_Client
{
	/**
	* Output type - response or call
	* @var string
	*/
	private $outputtype = null;

	/**
	* vBulletin Registry Object
	*
	* @var	Object
	*/
	private $registry = null;

	/**
	* vBulletin XML Object
	*
	* @var	Object
	*/
	private $xml_object = null;

	/**
	 * @var array
	 */
	private $options = null;

	/**
	* Constructor
	*
	* @param	object	vBulletin Registry Object
	*/
	function __construct(&$registry)
	{
		if (is_object($registry))
		{
			$this->registry =& $registry;
		}
		else
		{
			trigger_error(get_class($this) . '::Registry object is not an object', E_USER_ERROR);
		}

		$this->options = unserialize($this->registry->options['vbsoccer_sd']);
	}

	/**
	* Build XMPRPC Call Output
	*
	* The first parameter is the methodName
	* then (unlimited number of) following parameters are the params
	* Params can be sent as strings or arrays that define their type.
	*
	* @param	string	Text of the phrase
	* @param	mixed	First variable to be inserted
	* @param	mixed	Nth variable to be inserted
	*/
	private function _build_xml_call()
	{
		$args = func_get_args();
		$numargs = sizeof($args);

		if ($numargs < 1 OR !is_string($args[0]))
		{
			trigger_error('vBSoccer_XMLRPC_Client::_build_xml_call() Must specify a method (string)', E_USER_ERROR);
		}

		$this->_build_xml('methodCall', $args);
	}

	/**
	* Output the XML-RPC Call via HTTP POST
	*
	*/
	private function _send_xml_call($url)
	{
		if ($this->outputtype != 'call')
		{
			trigger_error('vBSoccer_XMLRPC_Client::_send_xml_call() Must call _build_xml_call() before _send_xml_call()', E_USER_ERROR);
		}

		$vurl = new vB_vURL($this->registry);
		$vurl->set_option(VURL_URL, $url);
		$vurl->set_option(VURL_POST, 1);
		$vurl->set_option(VURL_HEADER, 1);
		$vurl->set_option(VURL_ENCODING, 'gzip');
		$vurl->set_option(VURL_HTTPHEADER, array($this->xml_object->fetch_content_type_header(), sprintf('Authorization: Basic %s', $this->options['u'])));
		$vurl->set_option(VURL_USERAGENT, 'vBulletin/vBSoccer ' . $this->registry->options['bburl']);
		$vurl->set_option(VURL_MAXREDIRS, 1);
		$vurl->set_option(VURL_FOLLOWLOCATION, 1);
		$vurl->set_option(VURL_POSTFIELDS, $this->xml_object->fetch_xml_tag() . $this->xml_object->output());
		$vurl->set_option(VURL_RETURNTRANSFER, 1);
		$vurl->set_option(VURL_CLOSECONNECTION, 1);
		return $vurl->exec();
	}

	/**
	 *
	 */
	public function __call($method, $params=false)
	{
		if (!empty($params))
		{
			$this->_build_xml_call(sprintf('vbsoccer.%s', $method), $params);
		}
		else
		{
			$this->_build_xml_call(sprintf('vbsoccer.%s', $method));
		}

		$result = $this->_send_xml_call(sprintf('%srpc.php', $this->options['s']));

		$xml_object = new vB_XML_Parser($result['body']);
		$xml_object->include_first_tag = false;

		$return = array();

		if ($xml_object->parse_xml())
		{
			// The body of the response is a single XML structure,
			// a <methodResponse>, which can contain a single <params> which
			// contains a single <param> which contains a single <value>.
			// The <methodResponse> could also contain a <fault> which contains
			// a <value> which is a <struct> containing two elements, one named
			// <faultCode>, an <int> and one named <faultString>, a <string>.

			if (isset($xml_object->parseddata['fault']))
			{
				$faultCode = 0;
				$faultString = 'Unknown Exception!';
				$exception = $xml_object->parseddata['fault']['value']['struct']['member'];

				foreach ($exception AS $key => &$value)
				{
					if ($value['name'] == 'faultCode')
					{
						$faultCode = $value['value']['int'];
						continue;
					}

					if ($value['name'] == 'faultString')
					{
						$faultString = $value['value']['string'];
						continue;
					}
				}
				throw new Exception($faultString, $faultCode);
			}

			if (isset($xml_object->parseddata['params']['param']['value']))
			{
				$response =& $xml_object->parseddata['params']['param']['value'];
			}

			if (isset($response['boolean']))
			{
				$return = (bool) $response['boolean'];
			}

			if (isset($response['string']))
			{
				$return = $response['string'];
			}

			if (isset($response['array']['data']['value']))
			{
				if (count($response['array']['data']['value']) == 1)
				{
					$obj = unserialize(base64_decode($response['array']['data']['value']['base64']));
					$return[] = (array) $obj;
				}
				else
				{
					foreach ($response['array']['data']['value'] AS &$val)
					{
						$obj = unserialize(base64_decode($val['base64']));
						$return[] = (array) $obj;
					}
				}
			}
		}

		$xml_object = null;
		return $return;
	}

	/**
	* Private
	* Build XMPRPC Output
	*
	* The first parameter is the type, methodCall, methodResponse, or Fault (methodResponse)
	* then (unlimited number of) following parameters are the params
	* Params can be sent as strings or arrays that define their type.
	*
	* Per spec, methodResponse should have a maximum of one param
	*
	* @param	string	Type
	* @param	mixed	First variable to be inserted
	* @param	mixed	Nth variable to be inserted
	*
	*/
	private function _build_xml($type = 'methodCall')
	{
		$tempargs = func_get_args();
		$args     = $tempargs[1];

		$this->xml_object = new vB_XML_Builder($this->registry, null, 'UTF-8');

		// Empty doc in case we call this method multiple times
		$this->xml_object->doc = '';

		if ($type == 'methodCall')
		{
			$this->outputtype = 'call';
			$this->xml_object->add_group('methodCall');
			$this->xml_object->add_tag('methodName', $args[0]);
			array_shift($args);
		}
		else if ($type == 'methodResponse' OR $type == 'fault')
		{
			$this->outputtype = 'response';
			$this->xml_object->add_group('methodResponse');

			if ($type == 'fault')
			{
				$this->xml_object->add_group('fault');
				$this->_add_value($args[0]);
				$this->xml_object->close_group('fault');
				$this->xml_object->close_group('methodResponse');
				return;
			}
		}

		$this->xml_object->add_group('params');

		foreach($args AS $key => $value)
		{
			$this->xml_object->add_group('param');
			$this->_add_value($value);
			$this->xml_object->close_group('param');
		}

		$this->xml_object->close_group('params');

		if ($type == 'methodCall')
		{
			$this->xml_object->close_group('methodCall');
		}
		else
		{
			$this->xml_object->close_group('methodResponse');
		}
	}

	/**
	* Add <value> object to Output
	*
	* @param	string $value
	*/
	private function _add_value($value)
	{
		if (!is_array($value))
		{
			// convert string into an array
			$value = array('string' => $value);
		}

		$key = array_pop(array_keys($value));
		$this->xml_object->add_group('value');
		switch(strtolower($key))
		{
			case 'i4':
			case 'int':
				// Integer
				$this->xml_object->add_tag('int', intval($value["$key"]));
				break;
			case 'boolean':
				// boolean
				$this->xml_object->add_tag('boolean', ($value["$key"] == 1 OR strtolower($value["$key"]) == 'true') ? 1 : 0);
				break;
			case 'double':
				// float
				$this->xml_object->add_tag('double', floatval($value["$key"]));
				break;
			case 'datetime.iso8601':
				// datetime
				$this->xml_object->add_tag('dateTime.iso8601', strval($value["$key"]));
				break;
			case 'base64':
				// base64 encoded field
				// must already be encoded
				$this->xml_object->add_tag('base64', strval($value["$key"]));
				break;
			case 'array':
				if (!is_array($value["$key"]) OR empty($value["$key"]))
				{	// treat this as a string?
					$this->xml_object->add_tag('string', strval($value["$key"]));
				}
				else
				{
					$this->xml_object->add_group('array');
					$this->xml_object->add_group('data');
					foreach($value["$key"] AS $subkey => $subvalue)
					{
						$this->_add_value($subvalue);
					}
					$this->xml_object->close_group('data');
					$this->xml_object->close_group('array');
				}
				// array'
				break;
			case 'struct':
				// struct
				if (!is_array($value["$key"]) OR empty($value["$key"]))
				{	// treat this as a string?
					$this->xml_object->add_tag('string', strval($value["$key"]));
				}
				else
				{
					$this->xml_object->add_group('struct');
					foreach($value["$key"] AS $subkey => $subvalue)
					{
						$this->xml_object->add_group('member');
						$this->xml_object->add_tag('name', $subvalue['name']);
						unset($subvalue['name']);
						$this->_add_value($subvalue);
						$this->xml_object->close_group('member');
					}
					$this->xml_object->close_group('struct');
				}
				break;
			case 'string':
			default:
				$this->xml_object->add_tag('string', strval($value["$key"]));
		}
		$this->xml_object->close_group('value');
	}
}

/**
 * vbsoccer_Services
 *
 * @package vbsoccer
 */
class vbsoccer_Services
{
	public $stop_livescore_cron;

	protected $registry;

	protected $targetEncoding = 'WINDOWS-1252';

	protected $utf8DecodeResponses = false;

	protected $client;

	protected $sdMatchInfoMap = array(
		'id'     => 'sd_id',
		'dt'     => 'dateline',
		'lu'     => 'sd_lastupdate',
		't1'     => 'team_home_id',
		't2'     => 'team_away_id',
		'mdphid' => 'matchtype_id',
		'mdph'   => 'phrase_id',
		'md'     => 'match_day',
		's1'     => 'points_home',
		's2'     => 'points_away',
		'fi'     => 'match_is_finished',
		);

	protected $sdLeagueInfoMap = array(
		'sd_homepage'       => 'hp',
		'sd_relegation'     => 'rel',
		'sd_sent_back_down' => 'down',
		'sd_called_up'      => 'up',
		'mdgrp'             => 'mdgrp',
		'mdp'               => 'mdp',
		'showtable'         => 'tab',
		'region'            => 'reg',
		);

	/**
	 * @since 1.4.13
	 * @var array
	 */
	private $_lastupdateinfos;

	/**
	 * @since 1.4.13
	 * @var array
	 */
	private $_leagueinfos;

	/**
	 * @since 1.4.13
	 * @var string vbsoccer_livedata|vbsoccer_fixtures
	 */
	private $_fetch_type;

	/**
	 * @since 1.4.13
	 * @var bool
	 */
	private $_rebuild_language;

	/**
	 * @since 1.4.13
	 * @var bool
	 */
	private $_rebuild_leagues_datastore;

	/**
	 * @since 1.4.13
	 * @var array
	 */
	private $_rebuild_league_tables;

	/**
	 * @since 1.4.13
	 * @var bool
	 */
	private $_remove_forumblock_cache;

	/**
	 * constructor
	 *
	 * @param object $registry
	 */
	public function __construct(&$registry)
	{
		$this->registry =& $registry;

		if (strtolower(vB_Template_Runtime::fetchStyleVar('charset')) != strtolower($this->targetEncoding))
		{
			$this->targetEncoding = vB_Template_Runtime::fetchStyleVar('charset');
		}

		if ('UTF-8' !== strtoupper($this->targetEncoding))
		{
			$this->utf8DecodeResponses = true;
		}

		$this->client = new vBSoccer_XMLRPC_Client($registry);
	}

	/**
	 * services ping
	 *
	 * methode ruft vbsoccer.ping im datendienst auf, welche mit dem string
	 * 'PONG!' antwortet. damit kann die verbindung zum datendienst getestet
	 * werden.
	 *
	 * @return string '(PONG|ERROR)!'
	 */
	public function ping()
	{
		try
		{
			return $this->client->ping();
		}
		catch (Exception $e)
		{
			return 'ERROR!';
		}
	}

	/**
	 * team-icon vom datendienst abrufen
	 *
	 * methode ruft vbsoccer.getTeamIcon im datendienst auf und erhaelt als
	 * antwort ein base64 encodiertes png-icon. der base64-string wird im acp
	 * direkt zur anzeige eingebunden und kann wahlweise als binaeres file auf
	 * dem server gespeichert werden ('upload'-funktion).
	 *
	 * @param int $id datendienst-id des teams
	 * @return string
	 */
	public function getTeamIcon($id=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getTeamIcon($id));
		}
		catch (Exception $e)
		{
			return '';
		}
	}

	/**
	 * liga-icon vom datendienst abrufen
	 *
	 * ruft im datendienst die methode vbsoccer.getLeagueIcon auf
	 *
	 * @see getTeamIcon()
	 * @param int $id datendienst-id der liga
	 * @return string
	 */
	public function getLeagueIcon($id=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getLeagueIcon($id));
		}
		catch (Exception $e)
		{
			return '';
		}
	}

	/**
	 * aktuelle liga-informationen vom datendienst abrufen
	 *
	 * ruft im datendienst die methode vbsoccer.getLeagues auf und liefert ein
	 * array mit liga-informationen. wird verwendet zur import-auswahl im acp
	 * und zum abgleich von bereits importierten liga-daten.
	 *
	 * @param int $leagueid optionale datendienst-id der angeforderten liga.
	 *                      ist der parameter 0 oder wird nicht angegeben,
	 *                      werden saemtliche verfuegbare ligen abgeholt.
	 * @return array
	 */
	public function getLeagues($leagueid=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getLeagues($leagueid), $leagueid);
		}
		catch (Exception $e)
		{
			return array();
		}
	}

	/**
	 * team-informationen vom datendienst abrufen
	 *
	 * ruft im datendienst die methode vbsoccer.getTeams auf und liefert die
	 * entsprechenden team-daten zurueck.
	 *
	 * @param int $leagueid datendienst-id einer liga. wird der parameter an
	 *                      den datendienst weitergegeben, werden die zur liga
	 *                      und saison gehoerigen teams nebst liga-spezifischen
	 *                      informationen geliefert (zb strafpunkte in saison).
	 * @return array
	 */
	public function getTeams($leagueid=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getTeams($leagueid));
		}
		catch (Exception $e)
		{
			return array();
		}
	}

	/**
	 * name einer liga vom datendienst abfragen
	 *
	 * ruft im datendienst die methode vbsoccer.getLeagueName auf und liefert
	 * den namen der angeforderten liga. diese funktion wird benoetigt, wenn in
	 * der vbulletin-umgebung masterphrasen der ligen geloescht werden.
	 *
	 * @param int $id datendienst-id einer liga
	 * @return string
	 */
	public function getLeagueName($id=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getLeagueName($id));
		}
		catch (Exception $e)
		{
			return '';
		}
	}

	/**
	 * name eines teams vom datendienst abfragen
	 *
	 * ruft im datendienst die methode vbsoccer.getTeamName auf und liefert
	 * den namen des angeforderten teams. diese funktion wird benoetigt, wenn in
	 * der vbulletin-umgebung masterphrasen der teams geloescht werden.
	 *
	 * @param int $id datendienst-id eines teams
	 * @return string
	 */
	public function getTeamName($id=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getTeamName($id));
		}
		catch (Exception $e)
		{
			return '';
		}
	}

	/**
	 * liefert zeitstempel der letzten aenderung der angeforderten ligen und
	 * zugehoerigen daten.
	 *
	 * ruft die methode vbsoccer.getLastupdateInfo im datendienst auf und
	 * liefert die zeitstempel von aenderungen der ligen, zugehoerigen teams und
	 * ansetzungen zum lokalen abgleich.
	 *
	 * @since 1.4.13
	 * @param array $leagues array mit datendienst-ids der ligen
	 * @return array
	 */
	public function getLastupdateInfo($leagues=array())
	{
		if (!empty($leagues) AND is_array($leagues))
		{
			$sleagues = serialize($leagues);
		}
		else
		{
			$sleagues = 0;
		}

		unset($leagues);

		try
		{
			return $this->_fetchResponseData($this->client->getLastupdateInfo($sleagues));
		}
		catch (Exception $e)
		{
			return array();
		}
	}

	/**
	 * datendienst-abruf von ansetzungen zu einer bestimmten liga
	 *
	 * ruft im datendienst die methode vbsoccer.getLeagueMatches auf und liefert
	 * saemtliche spiele einer saison zurueck.
	 *
	 * @param int $leagueid datendienst-id einer liga
	 * @return array
	 */
	public function getLeagueMatches($leagueid=0)
	{
		try
		{
			return $this->_fetchResponseData($this->client->getLeagueMatches($leagueid));
		}
		catch (Exception $e)
		{
			return array();
		}
	}

	/**
	 * datendienst-abruf von laufenden spiel-informationen ('live-scores')
	 *
	 * ruft im datendienst die methode vbsoccer.latestResults auf und liefert
	 * spielstandsinformationen zu den angegebenen spielen zurueck.
	 *
	 * @param array $matches datendienst-ids laufender spiele
	 * @return array
	 */
	public function latestResults($matches=array())
	{
		$params = new stdClass;
		$params->matches = array_keys($matches);
		$matches = serialize($params);
		unset($params);

		try
		{
			return $this->_fetchResponseData($this->client->latestResults($matches));
		}
		catch (Exception $e)
		{
			return array();
		}
	}

	/**
	 * sendet einen fehlerreport an den datendienst
	 *
	 * ruft im datendienst die methode vbsoccer.report auf und veranlasst die
	 * email-benachrichtigung der datendienstbetreiber ueber den datendienst.
	 *
	 * @param array $report informationen, die an den datendienst gesendet und
	 *              an die betreiber weitergereicht werden:
	 *
	 *              'rusername' aktueller benutzername des forumbenutzers.
	 *              'ruserid'   userid des forumbenutzers.
	 *                          diese angaben werden bei missbrauch der
	 *                          funktion fuer die meldung an den forenbetreiber
	 *                          benoetigt.
	 *              'remail'    emailadresse des forumbenutzers fuer weitere
	 *                          kommunikation mit dem user per email.
	 *              'reason'    eingegebener text des forumbenutzers.
	 *
	 *              die uebermittelten daten werden *nicht* im datendienst
	 *              gespeichert, sondern lediglich in einer email an die
	 *              betreiber des datendienstes gesendet!
	 *
	 * @return boolean
	 */
	public function report($report)
	{
		try
		{
			return $this->client->report(serialize($report));
		}
		catch (Exception $e)
		{
			return false;
		}
	}

	/**
	 * uebermittelt spielstaende an den datendienst
	 *
	 * ruft die methode vbsoccer.updateMatches im datendienst auf und sendet die
	 * vom forumbenutzer gemeldeten spielstandsaenderungen mit.
	 *
	 * @param array $matches informationen ueber spielstaende
	 * @return boolean
	 */
	public function updateMatches($matches)
	{
		try
		{
			return $this->client->updateMatches(serialize($matches));
		}
		catch (Exception $e)
		{
			return false;
		}
	}

	/**
	 * prueft spiele auf aenderungen zur spielzeit
	 *
	 * @params void
	 * @return boolean
	 */
	public function check_livescores()
	{
		$this->_fetch_type                = 'vbsoccer_livedata';
		$this->_rebuild_language          = false;
		$this->_rebuild_leagues_datastore = false;
		$this->_remove_forumblock_cache   = false;
		$this->_rebuild_league_tables     = array();
		$this->stop_livescore_cron        = false;

		if (!$this->_fetch_changed_timestamps())
		{
			return false;
		}

		$changedMatchesLeagues = $matchinfos = array();

		if (empty($this->_leagueinfos))
		{
			return false;
		}

		foreach ($this->_leagueinfos AS $leagueinfo)
		{
			$changedMatchesLeagues[] = $leagueinfo['id'];
			$lastupdateinfo = $this->_lastupdateinfos[$leagueinfo['sd_id']];
			//$leagueUpdateSql[$leagueinfo['id']]['sd_lastupdate_matches'] = sprintf("`sd_lastupdate_matches`=%d", $lastupdateinfo['lastupdate_matches']);
			// *** ACHTUNG ***
			// wird hier der tatsaechliche zeitstempel vom datendienst uebernommen,
			// werden spiele ausserhalb des zeitfensters fuer livespiele nicht
			// beruecksichtigt! TIMENOW ist quasi das erzwingen eines updates aller
			// spiele beim nachsten durchlauf des aktualisierungs-jobs.
			// es bedeutet aber auch, dass jeder durchlauf der livescore-scripte
			// alle ids des nachfolgenden sql-queries abholt
			$leagueUpdateSql[$leagueinfo['id']]['sd_lastupdate_matches'] = sprintf("`sd_lastupdate_matches`=%d", TIMENOW);
		}

		$matches = $this->registry->db->query_read("
			SELECT `id` AS matchid, `sd_id` AS id, `sd_lastupdate` AS lu, `dateline` AS dt,
				`points_home` AS s1, `points_away` AS s2, IF(`match_is_finished`=2, 0, `match_is_finished`) AS `fi`,
				`league_id`
			FROM `" . TABLE_PREFIX . "soccer_match`
			WHERE `dateline` BETWEEN " . (TIMENOW - 28800) . " AND " . (TIMENOW + 28800) . "
				AND `league_id` IN(" . implode(',', $changedMatchesLeagues) . ")
			");

		unset($changedMatchesLeagues);

		while (($match = $this->registry->db->fetch_array($matches)))
		{
			$matchinfos[$match['id']] = array_map('intval', $match);
		}

		$this->registry->db->free_result($matches);
		unset($matches);

		if (empty($matchinfos))
		{
			$this->stop_livescore_cron = true;
			return false;
		}

		if (!($fixtures = $this->latestResults($matchinfos)))
		{
			return false;
		}

		$postresultUpdateSql = array();

		foreach ($fixtures AS $id => $fixturesdata)
		{
			$fixturesdata = (array) $fixturesdata;

			if (!isset($matchinfos[$fixturesdata['id']]))
			{
				continue;
			}

			$matchinfo = $matchinfos[$fixturesdata['id']];

			if ($fixturesdata['lu'] == $matchinfo['lu'])
			{
				continue;
			}

			$sdmatchinfo = array(
				'lu' => $fixturesdata['lu'],
				's1' => $fixturesdata['s1'],
				's2' => $fixturesdata['s2'],
				'fi' => (int) $fixturesdata['fi'],
				'dt' => $fixturesdata['dt'],
				);

			if (($diff = array_diff_assoc($sdmatchinfo, $matchinfo)))
			{
				$matchUpdateSql = array();

				foreach ($diff AS $key => $value)
				{
					if ($key == 's1' OR $key == 's2')
					{
						if ($sdmatchinfo['fi'] == 0 AND $sdmatchinfo[$key] < $matchinfo[$key] AND $sdmatchinfo['dt'] < TIMENOW)
						{
							continue;
						}
					}

					$matchUpdateSql[$key] = sprintf('`%s`=%d', $this->sdMatchInfoMap[$key], $value);

					if ($key == 'fi')
					{
						if ($matchinfo['fi'] == 0 AND $sdmatchinfo['fi'] == 1)
						{
							$this->_rebuild_league_tables[(int) $matchinfo['league_id']] = $matchinfo['league_id'];
						}
					}
				}

				if ($sdmatchinfo['dt'] == VBSOCCER_DEFERRED)
				{
					// zeitstempel des spiels ist identisch mit dem alias fuer
					// unbekannte anstosszeit (spiel abgesagt, etc.)
					//
					// bereits eingetragene ergebnisse zuruecksetzen auf -1
					$matchUpdateSql['s1'] = sprintf('`%s`=-1', $this->sdMatchInfoMap['s1']);
					$matchUpdateSql['s2'] = sprintf('`%s`=-1', $this->sdMatchInfoMap['s2']);

					// und spiel auf 'nicht angepfiffen' setzen, es sei denn, es ist
					// im datendienst explizit so gesetzt (spiel findet nicht mehr
					// statt)
					$matchUpdateSql['fi'] = sprintf('`%s`=%d', $this->sdMatchInfoMap['fi'], $sdmatchinfo['fi']);
				}

				if (($sdmatchinfo['s1'] != -1 AND isset($matchUpdateSql['s1'])) OR
					($sdmatchinfo['s2'] != -1 AND isset($matchUpdateSql['s2'])))
				{
					// ergebnis-string des neuen ergebnisses fuer aenderungslog
					$postresultUpdateSql[] = sprintf("(0, %d, %d, '%d:%d', %d, '')", $matchinfo['matchid'], TIMENOW, $sdmatchinfo['s1'], $sdmatchinfo['s2'], $sdmatchinfo['fi']);
				}

				if (!empty($matchUpdateSql))
				{
					// zeitstempel der aktualisierung setzen
					$matchUpdateSql[] = "`last_update_timestamp`=" . TIMENOW . "";

					$this->registry->db->query_write("
						UPDATE `" . TABLE_PREFIX . "soccer_match`
						SET " . implode(',', $matchUpdateSql) . ",
							`score_key`=CASE
								WHEN `points_home`=-1 AND `points_away`=-1 THEN 0
								WHEN `points_home`=`points_away` THEN 2
								WHEN `points_home`>`points_away` THEN 1
								WHEN `points_home`<`points_away` THEN 3
							END
						WHERE `id`=" . (int) $matchinfo['matchid'] . "
						");

					unset($matchUpdateSql);
				}
			}
		} // endforeach $fixtures

		unset($fixtures, $matchinfos);

		if (!empty($postresultUpdateSql))
		{
			$this->registry->db->query("
				INSERT INTO `" . TABLE_PREFIX . "soccer_postresult_log`
					(`userid`, `match_id`, `dateline`, `result`, `finished`, `ipaddress`)
				VALUES " . implode(',', $postresultUpdateSql) . "
				");
			unset($postresultUpdateSql);
		}

		if (!empty($leagueUpdateSql))
		{
			foreach ($leagueUpdateSql AS $leagueid => $sql)
			{
				if (isset($this->_rebuild_league_tables[$leagueid]))
				{
					$sql['x_ranking'] = sprintf("`x_ranking`='%s'", $this->registry->db->escape_string(construct_saison_rankings($leagueid)));
					$this->_remove_forumblock_cache = true;
					build_leaguetable($leagueid);
				}

				$this->registry->db->query_write("
					UPDATE `" . TABLE_PREFIX . "soccer_league`
					SET " . implode(',', $sql) . "
					WHERE `id`=" . $leagueid . "
					");

				unset($sql);
				$this->_rebuild_leagues_datastore = true;
			}

			unset($leagueUpdateSql);
		}

		if ($this->_rebuild_leagues_datastore === true)
		{
			// datastore fuer ligainformationen neu erstellen
			build_leagues_datastore();
		}

		if ($this->_remove_forumblock_cache === true)
		{
			$this->registry->db->query_write("
				DELETE FROM `" . TABLE_PREFIX . "cache`
				WHERE `cacheid` LIKE 'forumblock.vbsoccer.%'
				");
		}
	}

	/**
	 *
	 */
	public function check_fixtures($league_sdid=0)
	{
		$this->_fetch_type                = 'vbsoccer_fixtures';
		$this->_rebuild_language          = false;
		$this->_rebuild_leagues_datastore = false;
		$this->_remove_forumblock_cache   = false;
		$this->_rebuild_league_tables     = array();

		if (!$this->_fetch_changed_timestamps($league_sdid))
		{
			return false;
		}

		foreach ($this->_leagueinfos AS $league_sdid => $leagueinfo)
		{
			$lastupdateinfo = $this->_lastupdateinfos[$league_sdid];

			$leagueUpdateSql = $leagueTeamPhraseUpdateSql = $leagueTeamUpdateSql =
				$matchtypeUpdateSql = $postresultUpdateSql = array();

			if ($lastupdateinfo['lastupdate_league'] != $leagueinfo['sd_lastupdate'])
			{
				/***************************************************************
				*
				* liga updates
				*
				***************************************************************/
				if (false === ($leagueUpdateSql = $this->_check_league($leagueinfo)))
				{
					continue;
				}
			}

			/*******************************************************************
			 *
			 * team und liga-team updates
			 *
			 ******************************************************************/
			if ($lastupdateinfo['lastupdate_teams'] != $leagueinfo['sd_lastupdate_teams'])
			{
				if (!$this->_check_league_teams($leagueinfo))
				{
					return false;
				}
				$leagueUpdateSql['tabledata'] = "`tabledata`=''";
				$leagueUpdateSql['sd_lastupdate_teams'] = sprintf("`sd_lastupdate_teams`=%d", $lastupdateinfo['lastupdate_teams']);
			}


			/*******************************************************************
			 *
			 * ansetzungen abgleichen
			 *
			 ******************************************************************/
			if ($lastupdateinfo['lastupdate_matches'] != $leagueinfo['sd_lastupdate_matches'])
			{
				$leagueUpdateSql['sd_lastupdate_matches'] = sprintf("`%s`=%d", 'sd_lastupdate_matches', $lastupdateinfo['lastupdate_matches']);

				$matches = $this->registry->db->query_read("
					SELECT
						m.`id`                AS `matchid`,
						m.`sd_id`             AS `id`,
						m.`sd_lastupdate`     AS `lu`,
						m.`dateline`          AS `dt`,
						m.`match_day`         AS `md`,
						m.`matchtype_id`      AS `mdphid`,
						m.`points_home`       AS `s1`,
						m.`points_away`       AS `s2`,
						IF(m.`match_is_finished`=2, 0, m.`match_is_finished`) AS `fi`,
						mt.`phrase_id`        AS `mdph`,
						team1.`sd_id`         AS `t1`,
						team2.`sd_id`         AS `t2`
					FROM `" . TABLE_PREFIX . "soccer_match` AS m
					LEFT JOIN `" . TABLE_PREFIX . "soccer_matchtype` AS mt USING(`matchtype_id`)
					LEFT JOIN `" . TABLE_PREFIX . "soccer_team` AS team1 ON(team1.id=m.`team_home_id`)
					LEFT JOIN `" . TABLE_PREFIX . "soccer_team` AS team2 ON(team2.id=m.`team_away_id`)
					WHERE m.`league_id`=" . $leagueinfo['id'] . "
					");

				while (($match = $this->registry->db->fetch_array($matches)))
				{
					$matchinfos[(int) $match['id']] = $match;
				}

				$this->registry->db->free_result($matches);
				unset($matches);

				// alle ansetzungen fuer liga holen
				if (!($sdmatchinfos = $this->getLeagueMatches((int) $league_sdid)))
				{
					continue;
				}

				// neue ansetzungen vorhanden?
				if (($diff = array_diff_assoc($sdmatchinfos, $matchinfos)))
				{
					foreach ($diff AS $diffdata)
					{
						// spiel einfuegen (nur die ids!)
						$this->registry->db->query_write("
							INSERT INTO `" . TABLE_PREFIX . "soccer_match`
								(`sd_id`, `league_id`)
							VALUES (" . (int) $diffdata['id'] . ", " . $leagueinfo['id'] . ")
							");

						$matchinfos[(int) $diffdata['id']] = array('matchid' => $this->registry->db->insert_id());
					}
				}

				// vorhandene spiele mit den spieldaten des datendienstes direkt vergleichen
				foreach ($sdmatchinfos AS $sdmatchinfo)
				{
					$matchUpdateSql = array();

					$sdmatchinfo['fi'] = (int) $sdmatchinfo['fi'];
					$matchinfo = $matchinfos[$sdmatchinfo['id']];

					if (($diff = array_diff_assoc($sdmatchinfo, $matchinfo)))
					{
						if (isset($diff['mdph']) AND isset($diff['mdphid']) AND !isset($matchtypeUpdateSql[(int) $diff['mdphid']]))
						{
							// eines aenderung des spieltagtypes wurde erkannt.
							$matchtypeUpdateSql[(int) $diff['mdphid']] = sprintf("('%s', %d)", $this->registry->db->escape_string($diff['mdph']), $diff['mdphid']);
						}

						if (isset($diff['mdph']))
						{
							unset($diff['mdph']);
						}

						foreach ($diff AS $key => $value)
						{
							if ($key == 't1' OR $key == 't2')
							{
								// hoppla, hier wurden teams einer ansetzung geaendert!
								if (($teamresult = $this->registry->db->query_first("
									SELECT `id` FROM `" . TABLE_PREFIX . "soccer_team` WHERE `sd_id`=" . (int) $value . "")))
								{
									$value = $teamresult['id'];
									unset($teamresult);
								}
								else
								{
									// das team ist lokal nicht vorhanden?
									// abbrechen mit dem update und alle zeitstempel
									// der liga aendern um ein komplett-update zu erzwingen
									$this->registry->db->query_write("
										UPDATE `" . TABLE_PREFIX . "soccer_league`
										SET `sd_lastupdate_matches`=" . TIMENOW . ",
											`sd_lastupdate_teams`=" . TIMENOW . ",
											`sd_lastupdate`=" . TIMENOW . "
										WHERE id=" . $leagueinfo['id'] . "
										");

									return false;
								}
							}

							if ($key == 's1' OR $key == 's2')
							{
								if ($sdmatchinfo['fi'] == 0 AND $sdmatchinfo[$key] < $matchinfo[$key] AND $sdmatchinfo['dt'] < TIMENOW)
								{
									continue;
								}
							}

							$matchUpdateSql[$key] = sprintf('`%s`=%d', $this->sdMatchInfoMap[$key], $value);

							if ($key == 'fi')
							{
								if ($matchinfo['fi'] == 0 AND $sdmatchinfo['fi'] == 1)
								{
									$this->_rebuild_league_tables[(int) $leagueinfo['id']] = $leagueinfo['id'];
								}
							}
						}

						if ($sdmatchinfo['dt'] == VBSOCCER_DEFERRED)
						{
							// zeitstempel des spiels ist identisch mit dem alias fuer
							// unbekannte anstosszeit (spiel abgesagt, etc.)
							//
							// bereits eingetragene ergebnisse zuruecksetzen auf -1
							$matchUpdateSql['s1'] = sprintf('`%s`=-1', $this->sdMatchInfoMap['s1']);
							$matchUpdateSql['s2'] = sprintf('`%s`=-1', $this->sdMatchInfoMap['s2']);

							// und spiel auf 'nicht angepfiffen' setzen, es sei denn, es ist
							// im datendienst explizit so gesetzt (spiel findet nicht mehr
							// statt)
							$matchUpdateSql['fi'] = sprintf('`%s`=%d', $this->sdMatchInfoMap['fi'], $sdmatchinfo['fi']);
						}

						if (($sdmatchinfo['s1'] != -1 AND isset($matchUpdateSql['s1'])) OR
							($sdmatchinfo['s2'] != -1 AND isset($matchUpdateSql['s2'])))
						{
							// ergebnis-string des neuen ergebnisses fuer aenderungslog
							$postresultUpdateSql[] = sprintf("(0, %d, %d, '%d:%d', %d, '')", $matchinfo['matchid'], TIMENOW, $sdmatchinfo['s1'], $sdmatchinfo['s2'], $sdmatchinfo['fi']);
						}
					}

					if (!empty($matchUpdateSql))
					{
						// zeitstempel der aktualisierung setzen
						$matchUpdateSql[] = "`last_update_timestamp`=" . TIMENOW . "";

						$this->registry->db->query_write("
							UPDATE `" . TABLE_PREFIX . "soccer_match`
							SET " . implode(',', $matchUpdateSql) . ",
								`score_key`=CASE
									WHEN `points_home`=-1 AND `points_away`=-1 THEN 0
									WHEN `points_home`=`points_away` THEN 2
									WHEN `points_home`>`points_away` THEN 1
									WHEN `points_home`<`points_away` THEN 3
								END
							WHERE `sd_id`=" . (int) $sdmatchinfo['id'] . "
							");

						unset($matchUpdateSql);
					}
				}

				unset($sdmatchinfos, $matchinfos);

				if (!empty($matchtypeUpdateSql))
				{
					$this->registry->db->query_write("
						INSERT IGNORE INTO `" . TABLE_PREFIX . "soccer_matchtype`
							(`phrase_id`, `matchtype_id`)
						VALUES " . implode(',', $matchtypeUpdateSql) . "
						");

					unset($matchtypeUpdateSql);
				}
			}

			if (!empty($postresultUpdateSql))
			{
				$this->registry->db->query("
					INSERT INTO `" . TABLE_PREFIX . "soccer_postresult_log`
						(`userid`, `match_id`, `dateline`, `result`, `finished`, `ipaddress`)
					VALUES " . implode(',', $postresultUpdateSql) . "
					");

				unset($postresultUpdateSql);
			}

			if (!empty($leagueUpdateSql))
			{
				if (isset($this->_rebuild_league_tables[$leagueinfo['id']]))
				{
					$leagueUpdateSql['x_ranking'] = sprintf("`x_ranking`='%s'", $this->registry->db->escape_string(construct_saison_rankings($leagueinfo['id'])));
					$this->_remove_forumblock_cache = true;
					build_leaguetable($leagueinfo['id']);
				}

				$this->registry->db->query_write("
					UPDATE `" . TABLE_PREFIX . "soccer_league`
					SET " . implode(',', $leagueUpdateSql) . "
					WHERE `id`=" . $leagueinfo['id'] . "
					");

				$this->_debug(sprintf('$leagueUpdateSql: <pre>%s</pre>', print_r($leagueUpdateSql, true)));

				unset($leagueUpdateSql);
				$this->_rebuild_leagues_datastore = true;
			}
		} // end foreach (ligen)

		if ($this->_rebuild_leagues_datastore === true)
		{
			// datastore fuer ligainformationen neu erstellen
			build_leagues_datastore();
		}

		if ($this->_rebuild_language === true)
		{
			/** language tools */
			require_once DIR . '/includes/adminfunctions_language.php';
			build_language();
		}

		if ($this->_remove_forumblock_cache === true)
		{
			$this->registry->db->query_write("
				DELETE FROM `" . TABLE_PREFIX . "cache`
				WHERE `cacheid` LIKE 'forumblock.vbsoccer.%'
				");
		}

		return true;
	}


	/***************************************************************************
	 * DEPRECATED!                                                             *
	 **************************************************************************/

	/**
	 * holt die zeitstempel der letzten liga-aktualisierung vom datendienst ab
	 *
	 * @deprecated since version 1.4.13
	 * @see getLastupdateInfo()
	 * @param array $leagues
	 * @return array
	 */
	public function getChangedLeagues($leagues=array())
	{
		return $this->getLastupdateInfo($leagues);
	}


	/***************************************************************************
	 * PRIVATE                                                                 *
	 **************************************************************************/


	private function _check_league_teams($leagueinfo)
	{
		// teaminfos der liga-teams holen (namensaenderungen, etc.)
		if (!($sdteaminfos = $this->getTeams((int) $leagueinfo['sd_id'])))
		{
			return false;
		}

		// alle teams ermitteln, die bereits vorhanden sind
		$teams = $this->registry->db->query_read("
			SELECT t.`sd_id`, t.`id`, t.`sd_homepage`, t.`sd_shortname`, lt.`points`, lt.league_id, phrase.text AS masterphrase
			FROM `" . TABLE_PREFIX . "soccer_team` AS t
			LEFT JOIN `" . TABLE_PREFIX . "soccer_league_team` AS lt ON (lt.`team_id`=t.`id` AND lt.league_id=" . $leagueinfo['id'] . ")
			LEFT JOIN `" . TABLE_PREFIX . "phrase` AS phrase ON(phrase.`languageid`=0 AND phrase.`fieldname`='vbsoccernames' AND varname=CONCAT('soccer_team_', t.sd_id))
			WHERE t.`sd_id` IN(" . implode(',', array_map('intval', array_keys($sdteaminfos))) . ")
			");

		$teaminfos = array();
		$leagueTeamPhraseUpdateSql = array();
		$leagueTeamUpdateSql = array();

		while (($row = $this->registry->db->fetch_array($teams)))
		{
			$teaminfos[$row['sd_id']] = $row;
		}

		$this->registry->db->free_result($teams);
		unset($teams);

		foreach ($sdteaminfos AS $team_sdid => $sdteaminfo)
		{
			if (!isset($teaminfos[$team_sdid]))
			{
				$this->_debug(sprintf("Team %d ist nicht vorhanden und wird erstellt.", $team_sdid));

				// team existiert noch nicht
				$this->registry->db->query_write("
					INSERT INTO `" . TABLE_PREFIX . "soccer_team`
					SET `sd_id`       = " . (int) $team_sdid . ",
						`sd_shortname`='" . $this->registry->db->escape_string($sdteaminfo['sn']) . "',
						`sd_homepage` ='" . $this->registry->db->escape_string($sdteaminfo['hp']) . "'
					");

				$teaminfo = array('id' => $this->registry->db->insert_id());

				$params = array(
					$team_sdid,
					$this->registry->db->escape_string($sdteaminfo['name']),
					TIMENOW,
					$this->registry->db->escape_string($this->registry->options['templateversion'])
					);

				$leagueTeamPhraseUpdateSql[] = vsprintf("(0, 'vbsoccernames', 'soccer_team_%d', '%s', 'vbsoccer', 'soccerserv', %d, '%s')", $params);
				$leagueTeamUpdateSql[] = sprintf('(%d, %d, %d)', $leagueinfo['id'], $teaminfo['id'], $sdteaminfo['p']);
			}
			else
			{
				$teaminfo = $teaminfos[$team_sdid];
				$teamUpdateSql = array();

				if ($sdteaminfo['sn'] != $teaminfo['sd_shortname'])
				{
					// der kurzname des teams hat sich geaendert
					$teamUpdateSql[] = "`sd_shortname`='" . $this->registry->db->escape_string($sdteaminfo['sn']) . "'";
				}

				if ($sdteaminfo['hp'] != $teaminfo['sd_homepage'])
				{
					// die homepage des teams hat sich geaendert
					$teamUpdateSql[] = "`sd_homepage`='" . $this->registry->db->escape_string($sdteaminfo['hp']) . "'";
				}

				if (!empty($teamUpdateSql))
				{
					$this->_debug(sprintf("Team '%s' wird aktualisiert", $sdteaminfo['name']));
					// team aktualisieren, wenn aenderungen erkannt wurden
					$this->registry->db->query_write("
						UPDATE `" . TABLE_PREFIX . "soccer_team`
						SET " . implode(',', $teamUpdateSql) . "
						WHERE `id`=" . $teaminfo['id'] . "
						");
					unset($teamUpdateSql);
				}

				if ($sdteaminfo['p'] != $teaminfo['points'] OR empty($teaminfo['league_id']))
				{
					// strafpunkte des teams in dieser liga haben sich geaendert
					$leagueTeamUpdateSql[] = sprintf("(%d, %d, %d)", $leagueinfo['id'], $teaminfo['id'], $sdteaminfo['p']);
				}

				if ($sdteaminfo['name'] != $teaminfo['masterphrase'])
				{
					$params = array(
						$team_sdid,
						$this->registry->db->escape_string($sdteaminfo['name']),
						TIMENOW,
						$this->registry->db->escape_string($this->registry->options['templateversion'])
						);

					$leagueTeamPhraseUpdateSql[] = vsprintf("(0, 'vbsoccernames', 'soccer_team_%d', '%s', 'vbsoccer', 'soccerserv', %d, '%s')", $params);
				}
			}
		} // end foreach $sdteaminfos



		if (!empty($leagueTeamPhraseUpdateSql))
		{
			$this->_debug(sprintf("Master-Phrasen werden geaendert/eingefuegt"));
			$this->registry->db->query_write("
				REPLACE INTO `" . TABLE_PREFIX . "phrase`
					(`languageid`, `fieldname`, `varname`, `text`, `product`, `username`, `dateline`, `version`)
				VALUES " . implode(',', $leagueTeamPhraseUpdateSql) . "
				");
			$this->_rebuild_language = true;
			unset($leagueTeamPhraseUpdateSql);
		}

		if (!empty($leagueTeamUpdateSql))
		{
			$this->_debug(sprintf("Liga-Team-Beziehungen werden geaendert/eingefuegt"));
			// teams der liga zuordnen
			$this->registry->db->query_write("
				REPLACE INTO `" . TABLE_PREFIX . "soccer_league_team`
					(`league_id`, `team_id`, `points`)
				VALUES " . implode(',', $leagueTeamUpdateSql) . "
				");
			unset($leagueTeamUpdateSql);
		}

		return true;
	}

	/**
	 *
	 */
	private function _check_league($leagueinfo)
	{
		if (!($sdleagueinfo = $this->getLeagues($leagueinfo['sd_id'])))
		{
			return false;
		}

		$sdleagueinfo['tab'] = (int) $sdleagueinfo['tab'];

		// liganamen koennen das format "^name saison$" oder "name" haben.
		// basisname extrahieren (saisonangaben entfernen)
		$leaguetypename = preg_replace(array('/\/\d{4}/', '/\d{4}/'), array('', ''), $sdleagueinfo['name']);

		$leaguetype = strtolower($leagueinfo['sd_shortname']);

		if (($phraseinfo = $this->registry->db->query_first("
			SELECT `text`
			FROM `" . TABLE_PREFIX . "phrase`
			WHERE `languageid`=0 AND `fieldname`='vbsoccernames' AND
				`varname`='" . $this->registry->db->escape_string('soccer_leaguetype_' . $leaguetype) . "'
			")))
		{
			if (trim($phraseinfo['text']) != trim($leaguetypename))
			{
				$this->registry->db->query_write("
					UPDATE `" . TABLE_PREFIX . "phrase`
					SET `text`='" . $this->registry->db->escape_string(trim($leaguetypename)) . "'
					WHERE `languageid`=0 AND `fieldname`='vbsoccernames' AND
						`varname`='" . $this->registry->db->escape_string('soccer_leaguetype_' . $leaguetype) . "'
					");
				$this->_debug(sprintf("Ligatyp-Phrase aktualisiert"));
				$this->_rebuild_language = true;
			}
			unset($phraseinfo);
		}
		else
		{
			// TODO: phrase erstellen?
		}

		// ueberschreiben aller geaenderten sd_daten
		$leagueUpdateSql = array(
			'sd_lastupdate' => sprintf("`%s`=%d", 'sd_lastupdate', $lastupdateinfo['lastupdate_league'])
			);

		foreach ($this->sdLeagueInfoMap AS $key => $sdkey)
		{
			if ($sdleagueinfo[$sdkey] != $leagueinfo[$key])
			{
				$leagueUpdateSql[$key] = sprintf("`%s`='%s'", $key, $this->registry->db->escape_string($sdleagueinfo[$sdkey]));
			}
		}

		return $leagueUpdateSql;
	}

	/**
	 * abholen der aenderungszeitstempel vom datendienst
	 *
	 * die funktion setzt die eigenschaften von _lastupdateinfos und
	 * _leagueinfos vor weiteren pruefungen und abfragen.
	 *
	 * @since 1.4.13
	 * @param int $league_sdid liga-id im datendienst
	 * @return boolean
	 */
	private function _fetch_changed_timestamps($league_sdid=0)
	{
		$this->_lastupdateinfos = array();
		$this->_leagueinfos     = array();

		$dt = new DateTime();

		// ermitteln der turniere mit aktuellem jahr im saison-string
		$leagues = $this->registry->db->query_read("
			SELECT `id`, `sd_id`, `sd_lastupdate`, `last_cronjob_timestamp`,
				`sd_homepage`, `sd_relegation`, `sd_sent_back_down`, `region`,
				`sd_called_up`, `mdgrp`, `showtable`, `sd_lastupdate_matches`,
				`sd_lastupdate_teams`, `mdp`
			FROM `" . TABLE_PREFIX . "soccer_league`
			WHERE visible=1 AND `sd_saison` LIKE '%" . $dt->format('Y') . "%'
				" . ($league_sdid ? sprintf('AND `sd_id`=%d', $league_sdid) : "") . "
			");

		while (($league = $this->registry->db->fetch_array($leagues)))
		{
			$this->_leagueinfos[(int) $league['sd_id']] = $league;
		}

		$this->registry->db->free_result($leagues);
		unset($leagues, $dt);

		if (empty($this->_leagueinfos))
		{
			return false;
		}

		if (!($this->_lastupdateinfos = $this->getLastupdateInfo(array_keys($this->_leagueinfos))))
		{
			return false;
		}

		foreach ($this->_leagueinfos AS $league_sdid => $leagueinfo)
		{
			if (!isset($this->_lastupdateinfos[$league_sdid]))
			{
				unset($this->_leagueinfos[$league_sdid]);
				continue;
			}

			// TODO: pruefen, ob das casting hier wirklich noetig ist
			$this->_lastupdateinfos[$league_sdid] = (array) $this->_lastupdateinfos[$league_sdid];

			if ($this->_fetch_type == 'vbsoccer_livedata')
			{
				if ($this->_lastupdateinfos[$league_sdid]['lastupdate_matches'] == $this->_leagueinfos[$league_sdid]['sd_lastupdate_matches'])
				{
					// keine aenderung der spiele fuer diese liga
					unset($this->_leagueinfos[$league_sdid]);
				}

				continue;
			}

			if ($this->_lastupdateinfos[$league_sdid]['lastupdate_league'] == $this->_leagueinfos[$league_sdid]['sd_lastupdate'] AND
				$this->_lastupdateinfos[$league_sdid]['lastupdate_matches'] == $this->_leagueinfos[$league_sdid]['sd_lastupdate_matches'] AND
				$this->_lastupdateinfos[$league_sdid]['lastupdate_teams'] == $this->_leagueinfos[$league_sdid]['sd_lastupdate_teams'])
			{
				// alle zeitstempel dieser liga sind unveraendert
				unset($this->_leagueinfos[$league_sdid]);
				continue;
			}
		}

		return true;
	}

	/**
	 * aufbereitung der datendienst-rueckgaben
	 *
	 * konvertierung von verschiedenen datentypen in array(s) zur direkten
	 * weiterverarbeitung im tippspiel.
	 *
	 * @param array $response
	 * @param int $requested optional nur rueckgabe eines einzelnen datensatzes
	 *                       anhand der datendienst-id
	 * @return array
	 */
	private function _fetchResponseData($response = array(), $requested = 0)
	{
		if (empty($response) OR !is_array($response))
		{
			return array();
		}

		$return = array();

		$isArrayOfObjects = $isArrayOfArrays = false;

		if (is_object($response[0]))
		{
			$isArrayOfObjects = true;
		}
		else if (is_array($response[0]))
		{
			$isArrayOfArrays = true;
		}

		foreach ($response AS &$data)
		{
			// array mit objekten
			if ($isArrayOfObjects)
			{
				if (empty($data->id))
				{
					// jedes objekt hat eine eigenschaft 'id'. hier ist was sauer!
					continue;
				}

				if ($requested != 0 AND (int) $data->id != (int) $requested)
				{
					// nicht angeforderten datensatz ignorieren!
					continue;
				}

				// typecasting object -> array
				$return[$data->id] = (array) $data;

				if ($this->utf8DecodeResponses)
				{
					// alle werte im array dekodieren
					array_walk($return[$data->id], array(&$this, '_decodeResponseData'));
				}
			}
			else if ($isArrayOfArrays)
			{
				if (empty($data['id']))
				{
					// jedes array hat einen schluessel 'id'. hier ist was sauer!
					continue;
				}

				if ($requested != 0 and (int) $data['id'] != (int) $requested)
				{
					// nicht angeforderten datensatz ignorieren!
					continue;
				}

				if ($this->utf8DecodeResponses)
				{
					array_walk($data, array(&$this, '_decodeResponseData'));
				}

				$return[$data['id']] = $data;
			}
		} // end foreach

		unset($response);

		if ($requested != 0 AND isset($return[$requested]))
		{
			return $return[$requested];
		}

		return $return;
	}

	private function _decodeResponseData(&$value, $key)
	{
		static $iconv;
		static $mb_convert_encoding;
		static $keysToConvert;

		if (!$keysToConvert)
		{
			$keysToConvert = array('name', 'sn', 'ln');
		}

		if (in_array($key, $keysToConvert) AND $value != '')
		{
			$iconv_passed = $mb_passed = false;

			if (!$iconv)
			{
				$iconv = function_exists('iconv') ? 1 : -1;
			}

			if ($iconv === 1 AND $encoded_data = iconv('UTF-8', $this->targetEncoding . '//TRANSLIT', $value))
			{
				$iconv_passed = true;
				$value = $encoded_data;
			}

			if (!$mb_convert_encoding)
			{
				$mb_convert_encoding = function_exists('mb_convert_encoding') ? 1 : -1;
			}

			if (!$iconv_passed AND $mb_convert_encoding === 1 AND $encoded_data = @mb_convert_encoding($value, $this->targetEncoding, 'UTF-8'))
			{
				$mb_passed = true;
				$value = $encoded_data;
			}

			if (!$iconv_passed AND !$mb_passed)
			{
				$value = utf8_decode($value);
			}
		}
	}

	/**
	 * ausgabe von informationen (entwicklung)
	 *
	 * @param string $msg auszugebende nachricht
	 * @param boolean $nl html-zeilenumbruch anhaengen
	 */
	private function _debug($msg, $nl=true)
	{
		if ((defined('VB_AREA') AND VB_AREA == 'AdminCP') OR (defined('IN_CONTROL_PANEL') AND IN_CONTROL_PANEL === true))
		{
			echo $msg . ($nl ? '<br />' : '') . "\n";
			flush();
		}
	}
}
