<?php
/*========================================================================*\
|| ###################################################################### ||
|| # Video-Directory 1.2.2 (functions_videotags.php)			# ||
|| # -------------------------------------------------------------------# ||
|| # Copyright 2007-2008 Julian Schuh. All Rights Reserved.		# ||
|| # 		With significant modification by Jason Axelrod.		# ||
|| # This file may not be redistributed in whole or significant part.	# ||
|| # -------------------------------------------------------------------# ||
|| # 									# ||
|| ###################################################################### ||
\*========================================================================*/

$phrasegroups = array(
	'video'
);

function add_tags_to_video($videoinfo, $taglist)
{
	global $vbulletin;

	$videoid = intval($videoinfo['videoid']);
	if (!$videoid)
	{
		return array();
	}

	$taglist = fetch_video_tags($videoinfo, $taglist, $errors);

	if (!$taglist)
	{
		return $errors;
	}

	insert_tags_video($videoinfo, $taglist);

	return $errors;
}

function split_tag_video($taglist)
{
	global $vbulletin;
	static $delimiters = array();

	if (empty($delimiters))
	{
		$delimiter_list = $vbulletin->options['videodirectory_tagdelimiter'];
		$delimiters = array(',');

		// match {...} segments as is, then remove them from the string
		if (preg_match_all('#\{([^}]*)\}#s', $delimiter_list, $matches, PREG_SET_ORDER))
		{
			foreach ($matches AS $match)
			{
				if ($match[1] !== '')
				{
					$delimiters[] = preg_quote($match[1], '#');
				}
				$delimiter_list = str_replace($match[0], '', $delimiter_list);
			}
		}

		// remaining is simple, space-delimited text
		foreach (preg_split('#\s+#', $delimiter_list, -1, PREG_SPLIT_NO_EMPTY) AS $delimiter)
		{
			$delimiters[] = preg_quote($delimiter, '#');
		}
	}

	return ($delimiters
		? preg_split('#(' . implode('|', $delimiters) . ')#', $taglist, -1, PREG_SPLIT_NO_EMPTY)
		: array($taglist)
	);
}

function fetch_video_tags($videoinfo, $taglist, &$errors, $evalerrors = true)
{
	global $vbulletin;
	static $tagbadwords, $taggoodwords;

	$errors = array();

	if (!is_array($taglist))
	{
		$taglist = split_tag_video($taglist);
	}

	if (!trim($videoinfo['taglist']))
	{
		$existing_tags = array();
	}
	else
	{
		// this will always be delimited by a comma
		$existing_tags = explode(',', trim($videoinfo['taglist']));
	}

	$existing_tags = array();
	$current_tags = $vbulletin->db->query_read("
		SELECT videotagvideo.*
		FROM " . TABLE_PREFIX . "videotagvideo AS videotagvideo
		WHERE videotagvideo.videoid = $videoinfo[videoid]
	");
	while ($tag = $vbulletin->db->fetch_array($current_tags))
	{
		$existing_tags[] = $tag['tagid'];
	}

	if ($vbulletin->options['videodirectory_maxtagspervid'] AND count($existing_tags) >= $vbulletin->options['videodirectory_maxtagspervid'])
	{
		$errors['videomax'] = $evalerrors ? fetch_error('video_maximum_tags') : 'video_maximum_tags';
		return array();
	}

	if ($vbulletin->options['videodirectory_maxtagspervid'] AND (count($existing_tags) + count($taglist)) > $vbulletin->options['videodirectory_maxtagspervid'])
	{
		$toomany = (count($existing_tags) + count($taglist)) - $vbulletin->options['videodirectory_maxtagspervid'];
		$errors['videotagmax'] = $evalerrors ? fetch_error('video_maximum_tags_x', $toomany, $vbulletin->options['videodirectory_maxtagspervid']) : array('video_maximum_tags_x', $toomany, $vbulletin->options['videodirectory_maxtagspervid']);
		return array();
	}

	if ($vbulletin->options['videodirectory_tagmaxlen'] <= 0 OR $vbulletin->options['videodirectory_tagmaxlen'] >= 100)
	{
		$vbulletin->options['videodirectory_tagmaxlen'] = 100;
	}

	$valid_raw = array();

	// stop words: too common
	require(DIR . '/includes/searchwords.php'); // get the stop word list; allow multiple requires

	// filter the stop words by adding custom stop words (tagbadwords) and allowing through exceptions (taggoodwords)
	if (!is_array($tagbadwords))
	{
		$tagbadwords = preg_split('/\s+/s', vbstrtolower($vbulletin->options['videodirectory_tagbadwords']), -1, PREG_SPLIT_NO_EMPTY);
	}

	if (!is_array($taggoodwords))
	{
		$taggoodwords = preg_split('/\s+/s', vbstrtolower($vbulletin->options['videodirectory_taggoodwords']), -1, PREG_SPLIT_NO_EMPTY);
	}

	// merge hard-coded badwords and tag-specific badwords
	$badwords = array_merge($badwords, $tagbadwords);

	foreach ($taglist AS $tagtext)
	{
		$tagtext = trim(preg_replace('#[ \r\n\t]+#', ' ', $tagtext));
		if ($tagtext === '')
		{
			continue;
		}

		if (!in_array(vbstrtolower($tagtext), $taggoodwords))
		{
			$char_strlen = vbstrlen($tagtext, true);

			if ($vbulletin->options['videodirectory_tagminlen'] AND $char_strlen < $vbulletin->options['videodirectory_tagminlen'])
			{
				$errors['min_length'] = $evalerrors ? fetch_error('tag_too_short_min_x', $vbulletin->options['videodirectory_tagminlen']) : array('tag_too_short_min_x', $vbulletin->options['videodirectory_tagminlen']);
				continue;
			}

			if ($char_strlen > $vbulletin->options['videodirectory_tagmaxlen'])
			{
				$errors['max_length'] =  $evalerrors ? fetch_error('tag_too_long_max_x', $vbulletin->options['videodirectory_tagmaxlen']) : array('tag_too_long_max_x', $vbulletin->options['videodirectory_tagmaxlen']);
				continue;
			}

			if (strlen($tagtext) > 100)
			{
				// only have 100 bytes to store a tag
				$errors['max_length'] =  $evalerrors ? fetch_error('tag_too_long_max_x', $vbulletin->options['videodirectory_tagmaxlen']) : array('tag_too_long_max_x', $vbulletin->options['videodirectory_tagmaxlen']);
				continue;
			}

			$censored = fetch_censored_text($tagtext);
			if ($censored != $tagtext)
			{
				// can't have tags with censored text
				$errors['censor'] = $evalerrors ? fetch_error('tag_no_censored') : 'tag_no_censored';
				continue;
			}

			if (count(split_tag_video($tagtext)) > 1)
			{
				// contains a delimiter character
				$errors['comma'] = $evalerrors ? fetch_error('tag_no_comma') : 'tag_no_comma';
				continue;
			}

			if (in_array(strtolower($tagtext), $badwords))
			{
				$errors['common'] = $evalerrors ? fetch_error('tag_x_not_be_common_words', $tagtext) : array('tag_x_not_be_common_words', $tagtext);
				continue;
			}
		}

		$valid_raw[] = ($vbulletin->options['videodirectory_tagforcelower'] ? vbstrtolower($tagtext) : $tagtext);
	}

	// we need to essentially do a case-insensitive array_unique here
	$valid_unique = array_unique(array_map('vbstrtolower', $valid_raw));
	$valid = array();
	foreach (array_keys($valid_unique) AS $key)
	{
		$valid[] = $valid_raw["$key"];
	}
	$valid_unique = array_values($valid_unique); // make the keys jive with $valid

	if ($valid)
	{
		$existing_sql = $vbulletin->db->query_read("
			SELECT videotag.tagtext, IF(videotagvideo.tagid IS NULL, 0, 1) AS taginvideo
			FROM " . TABLE_PREFIX . "videotag AS videotag
			LEFT JOIN " . TABLE_PREFIX . "videotagvideo AS videotagvideo ON
				(videotag.tagid = videotagvideo.tagid AND videotagvideo.videoid = " . intval($videoinfo['videoid']) . ")
			WHERE videotag.tagtext IN ('" . implode("','", array_map(array(&$vbulletin->db, 'escape_string'), $valid)) . "')
		");

		$vbulletin->db->data_seek($existing_sql, 0);

		// determine which tags are already in the video and just ignore them
		while ($tag = $vbulletin->db->fetch_array($existing_sql))
		{
			if ($tag['taginvideo'])
			{
				// tag is in video, find it and remove
				if (($key = array_search(vbstrtolower($tag['tagtext']), $valid_unique)) !== false)
				{
					unset($valid["$key"], $valid_unique["$key"]);
				}
			}
		}
	}

	return $valid;
}

function insert_tags_video($video, $taglist)
{
	global $vbulletin;

	if (!$taglist OR !is_array($taglist))
	{
		return;
	}

	$taglist_db = array();
	$taglist_insert = array();
	foreach ($taglist AS $tag)
	{
		$tag = $vbulletin->db->escape_string($tag);

		$taglist_db[] = $tag;
		$taglist_insert[] = "('$tag', " . TIMENOW . ")";
	}

	// create new tags
	$vbulletin->db->query_write("
		INSERT IGNORE INTO " . TABLE_PREFIX . "videotag
			(tagtext, dateline)
		VALUES
			" . implode(',', $taglist_insert)
	);

	// now associate with video
	$tagvideo = array();
	$tagid_sql = $vbulletin->db->query_read("
		SELECT tagid
		FROM " . TABLE_PREFIX . "videotag AS videotag
		WHERE tagtext IN ('" . implode("', '", $taglist_db) . "')
	");
	while ($tag = $vbulletin->db->fetch_array($tagid_sql))
	{
		$tagvideo[] = "($video[videoid], $tag[tagid], " . $vbulletin->userinfo['userid'] . ", " . TIMENOW . ")";
	}

	if ($tagvideo)
	{
		// this should always happen
		$vbulletin->db->query_write("
			INSERT IGNORE INTO " . TABLE_PREFIX . "videotagvideo
				(videoid, tagid, userid, dateline)
			VALUES
				" . implode(',', $tagvideo)
		);
	}

	// now rebuild the tag list for the video
	rebuild_video_taglist($video);
}

function delete_video_tags(&$videoid)
{
	global $vbulletin;

	$vbulletin->db->query_write("
		DELETE FROM " . TABLE_PREFIX . "videotagvideo
		WHERE videoid = $videoid
		");
}

function rebuild_video_taglist($video, $limit = false)
{
	global $vbulletin;

	$tags = array();
	$tags_sql = $vbulletin->db->query_read("
		SELECT videotag.tagtext
		FROM " . TABLE_PREFIX . "videotagvideo AS videotagvideo
		INNER JOIN " . TABLE_PREFIX . "videotag AS videotag ON (videotag.tagid = videotagvideo.tagid)
		WHERE videotagvideo.videoid = $video[videoid]
		ORDER BY videotag.tagtext
	");
	while ($tag = $vbulletin->db->fetch_array($tags_sql))
	{
		$tags[] = '<a href="' . construct_tag_url($tag) . '">' . $tag['tagtext'] . '</a>';
	}

	$tagsize = count($tags);
	$taglist = implode(', ', $tags);

	if (!$tagsize)
	{
		$taglist = '<em>None...</em>';
	}

	$taglist2 = $taglist;
	if ($vbulletin->options['videodirectory_tagpreviewlen'] AND ($tagsize > $vbulletin->options['videodirectory_tagpreviewlen']))
	{
		$taglist2 = '<a href="' . construct_video_url($video) . '"><em>Too many (' . $tagsize . '), view details for full list...</em></a>';
	}

	$dm =& datamanager_init('Video', $vbulletin);
	$dm->set_existing($video);
	$dm->setr('cachetags', $taglist2);
	$dm->pre_save();
	$dm->save();

	return $taglist;
}

function construct_tag_url(&$tag)
{
	global $vbulletin;

	if (defined('SEO_ENABLED'))
	{
		global $vbulletin;
		$title = $tag['tagtext'];
		$title = strtolower($title);
		$title = unhtmlspecialchars($title);
		$title = str_replace("'", '', $title);
		$title = preg_split("#[^a-z0-9]#", $title, -1, PREG_SPLIT_NO_EMPTY);
		$title = array_slice($title, 0, 8);
		$title = implode("-",$title);
		$url = 'video-tag-' . $tag['tagtext'];
		unset($title);
	}
	else
	{
		$url = 'video.php?' . $vbulletin->session->vars['sessionurl'] . 'do=viewtag&amp;tag=' . $tag['tagtext'];
	}

	return $url;
}

function construct_video_cloud()
{
	global $vbulletin;

	$results = $vbulletin->db->query_read("
		SELECT videotagvideo.tagid AS tag, videotag.*, COUNT(*) AS quantity
			FROM " . TABLE_PREFIX . "videotagvideo AS videotagvideo
			INNER JOIN " . TABLE_PREFIX . "videotag AS videotag ON (videotag.tagid = videotagvideo.tagid)
			" . ($vbulletin->options['videodirectory_tagusage'] ?
				"WHERE videotagvideo.dateline > "	. (TIMENOW - (60 * 60 * 24 * $vbulletin->options['videodirectory_tagusage']))	:
				'') . "
			GROUP BY videotag.tagid, videotag.tagtext
			ORDER BY quantity DESC
			LIMIT " . $vbulletin->options['videodirectory_taglimit']
	);

	while ($row = $vbulletin->db->fetch_array($results))
	{
		$tags[$row['tag']] = $row['quantity'];
		$text[$row['tag']] = $row['tagtext'];
	}

	if (count($tags) == 0)
	{
		return 'None';
	}

	$max_size = $vbulletin->options['videodirectory_maxtagsize'];
	$min_size = $vbulletin->options['videodirectory_mintagsize'];

	$max_qty = max(array_values($tags));
	$min_qty = min(array_values($tags));
	
	$spread = $max_qty - $min_qty;

	if ($spread == 0)
	{
		$spread = 1;
	}

	$step = ($max_size - $min_size) / ($spread);

	asort($text);

	$taglist = array();
	
	foreach ($text as $key => $value)
	{
		$size = $min_size + (($tags[$key] - $min_qty) * $step);
		$size = floor($size);
		$taginfo['tagtext'] = $value;

		$taglist[] = '<span style="font-size:'.$size.'px" title="'.$tags[$key].' videos tagged with '.$value.'"><a href="'.construct_tag_url($taginfo).'">'.$value.'</a></span>';
	}

	$tagcloud = implode(' ', $taglist);

	return $tagcloud;
}

?>